/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.db;

import java.util.List;

import org.hibernate.HibernateException;
import org.hibernate.classic.Session;
import org.hibernate.criterion.Restrictions;
import org.jbpm.api.JbpmException;
import org.jbpm.api.ProcessEngine;
import org.jbpm.api.cmd.Command;
import org.jbpm.api.cmd.Environment;
import org.jbpm.internal.log.Log;
import org.jbpm.pvm.internal.cfg.ConfigurationImpl;
import org.jbpm.pvm.internal.id.PropertyImpl;
import org.jbpm.pvm.internal.processengine.ProcessEngineImpl;
import org.jbpm.pvm.internal.repository.DeploymentImpl;
import org.jbpm.pvm.internal.repository.DeploymentProperty;

/**
 * @author Alejandro Guizar
 */
public class Upgrade {

  private static final long serialVersionUID = 1L;
  
  private static Log log = Log.getLog(Upgrade.class.getName());
  
  static String database;

  public static void main(String[] args) {
    if ( (args==null)
         || (args.length!=1)
       ) {
      DbHelper.printSyntax(Upgrade.class);
      return;
    }
    
    database = args[0];
    
    ProcessEngine processEngine = new ConfigurationImpl()
      .skipDbCheck()
      .buildProcessEngine();
  
    try {
      JbpmVersion jbpmVersion = (JbpmVersion) processEngine.execute(new Command<Object>(){
        private static final long serialVersionUID = 1L;
        public Object execute(Environment environment) throws Exception {
          Session session = environment.get(Session.class);
          if (!PropertyImpl.propertiesTableExists(session)) {
            try {
              session.createSQLQuery("select CLASSNAME_ from JBPM4_VARIABLE").list();
              return JbpmVersion.V_4_1;

            } catch (HibernateException e) {
              return JbpmVersion.V_4_0;
            }
          }
    
          String dbVersion = PropertyImpl.getDbVersion(session);
          if (dbVersion == null) {
            throw new JbpmException("property table exists, but no db version property is present");
          }
    
          return JbpmVersion.getJbpmVersion(dbVersion);
        }
      });

      if (jbpmVersion == JbpmVersion.V_4_2) {
        throw new JbpmException("jBPM schema is already up to date");
      }

      if (jbpmVersion.isEarlier(JbpmVersion.V_4_1)) {
        processEngine.execute(new Command<Object>(){
          private static final long serialVersionUID = 1L;
          public Object execute(Environment environment) throws Exception {
            Session session = environment.get(Session.class);
            DbHelper.executeSqlResource("upgrade-4.0-to-4.1/jbpm." + database + ".upgrade.sql", session);
            return null;
          }
        });
      }

      if (jbpmVersion.isEarlier(JbpmVersion.V_4_2)) {
        processEngine.execute(new Command<Object>(){
          private static final long serialVersionUID = 1L;
          public Object execute(Environment environment) throws Exception {
            Session session = environment.get(Session.class);
            DbHelper.executeSqlResource("upgrade-4.1-to-4.2/jbpm." + database + ".upgrade.sql", session);
            PropertyImpl.upgradeProperties(session);
            return null;
          }
        });
        processEngine.execute(new Command<Object>(){
          private static final long serialVersionUID = 1L;
          public Object execute(Environment environment) throws Exception {
            Session session = environment.get(Session.class);
            // find deployments without a langid property
            List<DeploymentProperty> deploymentProperties = session.createCriteria(DeploymentProperty.class)
                .add(Restrictions.eq("key", DeploymentImpl.KEY_PROCESS_DEFINITION_ID))
                .list();

            for (DeploymentProperty deploymentProperty : deploymentProperties) {
              String objectName = deploymentProperty.getObjectName();
              DeploymentImpl deployment = deploymentProperty.getDeployment();
              deployment.setProcessLanguageId(objectName, "jpdl-4.0");
            }
            return null;
          }
        });
      }

      log.info("jBPM DB upgrade completed successfully.");

    } catch (Exception e) {
      log.error("ERROR: jBPM DB upgrade FAILED", e);

    } finally {
      processEngine.close();
    }
  }
}
