package ai.koog.a2a.model

import kotlinx.serialization.EncodeDefault
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonObject

/**
 * Message role.
 */
@Serializable
public enum class Role {
    @SerialName("user")
    User,

    @SerialName("agent")
    Agent
}

/**
 * Represents a single message in the conversation between a user and an agent.
 *
 * @property messageId A unique identifier for the message, typically a UUID, generated by the sender.
 * @property role Identifies the sender of the message. `user` for the client, `agent` for the service.
 * @property parts An array of content parts that form the message body. A message can be
 * composed of multiple parts of different types (e.g., text and files).
 * @property extensions The URIs of extensions that are relevant to this message.
 * @property taskId The ID of the task this message is part of. Can be omitted for the first message of a new task.
 * @property referenceTaskIds A list of other task IDs that this message references for additional context.
 * @property contextId The context ID for this message, used to group related interactions.
 * @property metadata Optional metadata for extensions. The key is an extension-specific identifier.
 */
@Serializable
public data class Message(
    public val messageId: String,
    public val role: Role,
    public val parts: List<Part>,
    public val extensions: List<String>? = null,
    public val taskId: String? = null,
    public val referenceTaskIds: List<String>? = null,
    public val contextId: String? = null,
    public val metadata: JsonObject? = null,
) : CommunicationEvent {
    @EncodeDefault
    override val kind: String = "message"
}
