package ai.koog.a2a.model

import kotlinx.datetime.Clock
import kotlinx.datetime.Instant
import kotlinx.serialization.EncodeDefault
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonObject

/**
 * Represents a single, stateful operation or conversation between a client and an agent.
 *
 * @property id A unique identifier (e.g. UUID) for the task, generated by the server for a new task.
 * @property contextId A server-generated unique identifier (e.g. UUID) for maintaining context across multiple related tasks or interactions.
 * @property status The current status of the task, including its state and a descriptive message.
 * @property history An array of messages exchanged during the task, representing the conversation history.
 * @property artifacts A collection of artifacts generated by the agent during the execution of the task.
 * @property metadata Optional metadata for extensions. The key is an extension-specific identifier.
 */
@Serializable
public data class Task(
    public val id: String,
    override val contextId: String,
    public val status: TaskStatus,
    public val history: List<Message>? = null,
    public val artifacts: List<Artifact>? = null,
    public val metadata: JsonObject? = null,
) : CommunicationEvent, TaskEvent {
    @EncodeDefault
    override val kind: String = "task"

    override val taskId: String get() = id
}

/**
 * Represents the status of a task at a specific point in time.
 *
 * @property state The current state of the task's lifecycle.
 * @property message An optional, human-readable message providing more details about the current status.
 * @property timestamp Timestamp indicating when this status was recorded.
 */
@Serializable
public data class TaskStatus(
    public val state: TaskState,
    public val message: Message? = null,
    public val timestamp: Instant? = Clock.System.now(),
)

/**
 * Defines the lifecycle states of a Task.
 *
 * @property terminal Indicates whether this is a terminal state.
 */
@Serializable
public enum class TaskState(public val terminal: Boolean) {
    /**
     * The task has been submitted and is awaiting execution.
     */
    @SerialName("submitted")
    Submitted(terminal = false),

    /**
     * The agent is actively working on the task.
     */
    @SerialName("working")
    Working(terminal = false),

    /**
     * The task is paused and waiting for input from the user.
     */
    @SerialName("input-required")
    InputRequired(terminal = false),

    /**
     * The task has been successfully completed.
     */
    @SerialName("completed")
    Completed(terminal = true),

    /**
     * The task has been canceled by the user.
     */
    @SerialName("canceled")
    Canceled(terminal = true),

    /**
     * The task failed due to an error during execution.
     */
    @SerialName("failed")
    Failed(terminal = true),

    /**
     *  The task was rejected by the agent and was not started.
     */
    @SerialName("rejected")
    Rejected(terminal = true),

    /**
     * The task requires authentication to proceed.
     */
    @SerialName("auth-required")
    AuthRequired(terminal = false),

    /*
     * The task is in an unknown or indeterminate state.
     */
    @SerialName("unknown")
    Unknown(terminal = false),
}
