// Copyright 2009 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package com.google.visualization.datasource.query.engine;

import com.google.common.collect.Maps;
import com.google.visualization.datasource.datatable.DataTable;
import com.google.visualization.datasource.datatable.TableRow;
import com.google.visualization.datasource.datatable.value.Value;
import com.google.visualization.datasource.query.AggregationType;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Aggregates a DataTable according to specific row groups. The groups are defined by an ordered
 * list of group-by columns. For instance, if the list is {"Name", "Revenue"} then each unique
 * pair of values of these columns (e.g., {"John", 300}, {"John", 19}, {"Sarah", 2222}) defines a
 * group, where the group includes all of the rows that share the values defined by the pair (for
 * example all the rows where value("Name") = "John" and value("Cost") = 300). In addition there is
 * a group for each unique name ({"John"}, {"Sarah"}) and an additional group that contains all the
 * rows in the table.
 * The groups described above are kept in an aggregation tree. The root of the tree contains
 * aggregation information for the group of all rows. Each level of the tree (except for the root
 * level) is associated with an aggregation column. Each node of the tree is associated with a
 * value (of some cell in the column defining the node's level). A path in the tree from the root
 * to a node is represented by an ordered list of values, and is associated with the group of data
 * rows identified by this list. In our example the tree contains one path of length 0 ({}), two
 * paths of length 1 ({"John"}, {"Sarah"}), and three paths of length two ({"John", 300},
 * {"John", 19}, {"Sarah", 2222}).
 * 
 * The aggregation data stored is all aggregation data possible for the columns to aggregate (also
 * called aggregation columns): the minimum, maximum, count, average, and sum, each of these
 * where applicable.
 *
 * @author Yoav G.
 */

public class TableAggregator {

  /**
   * An ordered list of columns to group by.
   */
  private List<String> groupByColumns;

  /**
   * A set of columns to aggregate.
   */
  private Set<String> aggregateColumns;

  /**
   * An aggregation tree is the logical data structure to use for grouping.
   */
  private AggregationTree tree;

  /**
   * Constructs a table aggregator and aggregates the table.
   *
   * @param groupByColumns An ordered list of columns to group by.
   * @param aggregateColumns A set of columns to aggregate.
   * @param table The table.
   */
  public TableAggregator(List<String> groupByColumns, Set<String> aggregateColumns,
      DataTable table) {

    this.groupByColumns = groupByColumns;
    this.aggregateColumns = aggregateColumns;

    tree = new AggregationTree(aggregateColumns, table);

    // Iterate over all rows and aggregate each row via the aggregation tree.
    for (TableRow row : table.getRows()) {
      tree.aggregate(getRowPath(row, table, groupByColumns.size() - 1), 
          getValuesToAggregate(row, table));
    }
  }

  /**
   * Creates a path for the aggregation tree defined by a table row.
   *
   * @param row The table row.
   * @param table The table.
   * @param depth The depth of the desired path.
   *
   * @return A path for the aggregation tree defined by the table row.
   */
  public AggregationPath getRowPath(TableRow row, DataTable table, int depth) {
    AggregationPath result = new AggregationPath();
    // The tree path is generated by looking for the values of the group-by
    // columns in the table row (in the correct order).
    for (int i = 0; i <= depth; i++) {
      String columnId = groupByColumns.get(i);
      Value curValue = row.getCell(table.getColumnIndex(columnId)).getValue();
      result.add(curValue);
    }
    return result;
  }

  /**
   * Returns a set containing the paths to all the leaves in the tree.
   *
   * @return A set containing the paths to all the leaves in the tree.
   */
  public Set<AggregationPath> getPathsToLeaves() {
    return tree.getPathsToLeaves();
  }

  /**
   * Creates a map from column id to value according to the aggregation columns.
   *
   * @param row The table row.
   * @param table The table.
   *
   * @return A map from column id to value according to the aggregation columns.
   */
  private Map<String, Value> getValuesToAggregate(TableRow row, DataTable table) {
    Map<String, Value> result = Maps.newHashMap();
    // The map is generated by looking for the values of the aggregation columns
    // in the table row.
    for (String columnId : aggregateColumns) {
      Value curValue = row.getCell(table.getColumnIndex(columnId)).getValue();
      result.put(columnId, curValue);
    }
    return result;
  }

  /**
   * Returns the aggregation value of a specific column and type.
   *
   * @param path The aggregation path.
   * @param columnId The requested column id.
   * @param type The requested aggregation type.
   *
   * @return The aggregation values of a specific column.
   */
  public Value getAggregationValue(AggregationPath path, String columnId,
      AggregationType type) {
    return tree.getNode(path).getAggregationValue(columnId, type);
  }
}
