/*
 * Decompiled with CFR 0.152.
 */
package com.hivemq.persistence.local.xodus;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.hivemq.bootstrap.ioc.lazysingleton.LazySingleton;
import com.hivemq.configuration.service.InternalConfigurations;
import com.hivemq.exceptions.UnrecoverableException;
import com.hivemq.extension.sdk.api.annotations.NotNull;
import com.hivemq.extension.sdk.api.annotations.Nullable;
import com.hivemq.extensions.iteration.BucketChunkResult;
import com.hivemq.migration.meta.PersistenceType;
import com.hivemq.persistence.PersistenceStartup;
import com.hivemq.persistence.RetainedMessage;
import com.hivemq.persistence.local.DeltaCounter;
import com.hivemq.persistence.local.rocksdb.RocksDBLocalPersistence;
import com.hivemq.persistence.local.xodus.PublishTopicTree;
import com.hivemq.persistence.local.xodus.RetainedMessageSerializer;
import com.hivemq.persistence.payload.PublishPayloadPersistence;
import com.hivemq.persistence.retained.RetainedMessageLocalPersistence;
import com.hivemq.util.LocalPersistenceFileUtil;
import com.hivemq.util.ThreadPreConditions;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.PostConstruct;
import javax.inject.Inject;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksIterator;
import org.rocksdb.WriteBatch;
import org.rocksdb.WriteOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@LazySingleton
public class RetainedMessageRocksDBLocalPersistence
extends RocksDBLocalPersistence
implements RetainedMessageLocalPersistence {
    private static final Logger log = LoggerFactory.getLogger(RetainedMessageRocksDBLocalPersistence.class);
    public static final String PERSISTENCE_VERSION = "040500_R";
    @VisibleForTesting
    @NotNull
    public final PublishTopicTree[] topicTrees;
    @NotNull
    private final PublishPayloadPersistence payloadPersistence;
    @NotNull
    private final AtomicLong retainMessageCounter = new AtomicLong(0L);

    @Inject
    public RetainedMessageRocksDBLocalPersistence(@NotNull LocalPersistenceFileUtil localPersistenceFileUtil, @NotNull PublishPayloadPersistence payloadPersistence, @NotNull PersistenceStartup persistenceStartup) {
        super(localPersistenceFileUtil, persistenceStartup, InternalConfigurations.PERSISTENCE_BUCKET_COUNT.get(), 32, 64, 32768, InternalConfigurations.RETAINED_MESSAGE_PERSISTENCE_TYPE.get() == PersistenceType.FILE_NATIVE);
        this.payloadPersistence = payloadPersistence;
        int bucketCount = this.getBucketCount();
        this.topicTrees = new PublishTopicTree[bucketCount];
        for (int i = 0; i < bucketCount; ++i) {
            this.topicTrees[i] = new PublishTopicTree();
        }
    }

    @Override
    @NotNull
    protected String getName() {
        return "retained_messages";
    }

    @Override
    @NotNull
    protected String getVersion() {
        return PERSISTENCE_VERSION;
    }

    @Override
    @NotNull
    protected Logger getLogger() {
        return log;
    }

    @Override
    @PostConstruct
    protected void postConstruct() {
        super.postConstruct();
    }

    @Override
    public void init() {
        try {
            DeltaCounter deltaCounter = DeltaCounter.finishWith(this.retainMessageCounter::addAndGet);
            for (int i = 0; i < this.buckets.length; ++i) {
                RocksDB bucket = this.buckets[i];
                PublishTopicTree publishTopicTree = this.topicTrees[i];
                try (RocksIterator iterator = bucket.newIterator();){
                    iterator.seekToFirst();
                    while (iterator.isValid()) {
                        RetainedMessage message = RetainedMessageSerializer.deserializeValue(iterator.value());
                        this.payloadPersistence.incrementReferenceCounterOnBootstrap(message.getPublishId());
                        String topic = RetainedMessageSerializer.deserializeKey(iterator.key());
                        publishTopicTree.add(topic);
                        deltaCounter.increment();
                        iterator.next();
                    }
                    continue;
                }
            }
            deltaCounter.run();
        }
        catch (Exception e) {
            log.error("An error occurred while preparing the Retained Message persistence.");
            log.debug("Original Exception:", (Throwable)e);
            throw new UnrecoverableException(false);
        }
    }

    @Override
    public void clear(int bucketIndex) {
        ThreadPreConditions.startsWith("single-writer");
        this.topicTrees[bucketIndex] = new PublishTopicTree();
        RocksDB bucket = this.buckets[bucketIndex];
        try (WriteBatch writeBatch = new WriteBatch();
             WriteOptions options = new WriteOptions();
             RocksIterator iterator = bucket.newIterator();){
            DeltaCounter retainMessageDelta = DeltaCounter.finishWith(this.retainMessageCounter::addAndGet);
            iterator.seekToFirst();
            while (iterator.isValid()) {
                RetainedMessage message = RetainedMessageSerializer.deserializeValue(iterator.value());
                this.payloadPersistence.decrementReferenceCounter(message.getPublishId());
                retainMessageDelta.decrement();
                writeBatch.delete(iterator.key());
                iterator.next();
            }
            bucket.write(options, writeBatch);
            retainMessageDelta.run();
        }
        catch (Exception e) {
            log.error("An error occurred while clearing the retained message persistence.");
            log.debug("Original Exception:", (Throwable)e);
        }
    }

    @Override
    public long size() {
        return this.retainMessageCounter.get();
    }

    @Override
    public void remove(@NotNull String topic, int bucketIndex) {
        Preconditions.checkNotNull((Object)topic, (Object)"Topic must not be null");
        ThreadPreConditions.startsWith("single-writer");
        RocksDB bucket = this.buckets[bucketIndex];
        try {
            byte[] key = RetainedMessageSerializer.serializeKey(topic);
            byte[] removed = bucket.get(key);
            if (removed == null) {
                log.trace("Removing retained message for topic {} (no message was stored previously)", (Object)topic);
                return;
            }
            RetainedMessage message = RetainedMessageSerializer.deserializeValue(removed);
            log.trace("Removing retained message for topic {}", (Object)topic);
            bucket.delete(key);
            this.topicTrees[bucketIndex].remove(topic);
            this.payloadPersistence.decrementReferenceCounter(message.getPublishId());
            this.retainMessageCounter.decrementAndGet();
        }
        catch (Exception e) {
            log.error("An error occurred while removing a retained message.");
            log.debug("Original Exception:", (Throwable)e);
        }
    }

    @Override
    @Nullable
    public RetainedMessage get(@NotNull String topic, int bucketIndex) {
        try {
            Preconditions.checkNotNull((Object)topic, (Object)"Topic must not be null");
            ThreadPreConditions.startsWith("single-writer");
            RocksDB bucket = this.buckets[bucketIndex];
            byte[] messageAsBytes = bucket.get(RetainedMessageSerializer.serializeKey(topic));
            if (messageAsBytes == null) {
                return null;
            }
            RetainedMessage message = RetainedMessageSerializer.deserializeValue(messageAsBytes);
            if (message.hasExpired()) {
                return null;
            }
            byte[] payload = this.payloadPersistence.get(message.getPublishId());
            if (payload == null) {
                log.warn("No payload was found for the retained message on topic {}.", (Object)topic);
                return null;
            }
            message.setMessage(payload);
            return message;
        }
        catch (Exception e) {
            log.error("An error occurred while getting a retained message.");
            log.debug("Original Exception:", (Throwable)e);
            return null;
        }
    }

    @Override
    public void put(@NotNull RetainedMessage retainedMessage, @NotNull String topic, int bucketIndex) {
        Preconditions.checkNotNull((Object)topic, (Object)"Topic must not be null");
        Preconditions.checkNotNull((Object)retainedMessage, (Object)"Retained message must not be null");
        ThreadPreConditions.startsWith("single-writer");
        RocksDB bucket = this.buckets[bucketIndex];
        try {
            byte[] serializedTopic = RetainedMessageSerializer.serializeKey(topic);
            byte[] valueAsBytes = bucket.get(serializedTopic);
            if (valueAsBytes != null) {
                RetainedMessage retainedMessageFromStore = RetainedMessageSerializer.deserializeValue(valueAsBytes);
                log.trace("Replacing retained message for topic {}", (Object)topic);
                bucket.put(serializedTopic, RetainedMessageSerializer.serializeValue(retainedMessage));
                this.payloadPersistence.decrementReferenceCounter(retainedMessageFromStore.getPublishId());
            } else {
                log.trace("Creating new retained message for topic {}", (Object)topic);
                bucket.put(serializedTopic, RetainedMessageSerializer.serializeValue(retainedMessage));
                this.topicTrees[bucketIndex].add(topic);
                this.retainMessageCounter.incrementAndGet();
            }
            this.payloadPersistence.add(retainedMessage.getMessage(), retainedMessage.getPublishId());
        }
        catch (Exception e) {
            log.error("An error occurred while persisting a retained message.");
            log.debug("Original Exception:", (Throwable)e);
        }
    }

    @Override
    @NotNull
    public Set<String> getAllTopics(@NotNull String subscription, int bucketId) {
        Preconditions.checkArgument((bucketId >= 0 && bucketId < this.getBucketCount() ? 1 : 0) != 0, (Object)"Bucket index out of range");
        ThreadPreConditions.startsWith("single-writer");
        return this.topicTrees[bucketId].get(subscription);
    }

    @Override
    public void cleanUp(int bucketId) {
        Preconditions.checkArgument((bucketId >= 0 && bucketId < this.getBucketCount() ? 1 : 0) != 0, (Object)"Bucket index out of range");
        ThreadPreConditions.startsWith("single-writer");
        if (this.stopped.get()) {
            return;
        }
        RocksDB bucket = this.buckets[bucketId];
        PublishTopicTree topicTree = this.topicTrees[bucketId];
        try (RocksIterator iterator = bucket.newIterator();
             WriteBatch writeBatch = new WriteBatch();
             WriteOptions options = new WriteOptions();){
            iterator.seekToFirst();
            while (iterator.isValid()) {
                String topic = RetainedMessageSerializer.deserializeKey(iterator.key());
                RetainedMessage message = RetainedMessageSerializer.deserializeValue(iterator.value());
                if (message.hasExpired()) {
                    writeBatch.delete(iterator.key());
                    this.payloadPersistence.decrementReferenceCounter(message.getPublishId());
                    this.retainMessageCounter.decrementAndGet();
                    topicTree.remove(topic);
                }
                iterator.next();
            }
            bucket.write(options, writeBatch);
        }
        catch (Exception e) {
            log.error("An error occurred while cleaning up retained messages.");
            log.debug("Original Exception:", (Throwable)e);
        }
    }

    @Override
    @NotNull
    public @NotNull BucketChunkResult<Map<String, @NotNull RetainedMessage>> getAllRetainedMessagesChunk(int bucketIndex, @Nullable String lastTopic, int maxMemory) {
        ThreadPreConditions.startsWith("single-writer");
        RocksDB bucket = this.buckets[bucketIndex];
        try (RocksIterator iterator = bucket.newIterator();){
            if (lastTopic == null) {
                iterator.seekToFirst();
            } else {
                String deserializedTopic;
                iterator.seek(RetainedMessageSerializer.serializeKey(lastTopic));
                if (iterator.isValid() && (deserializedTopic = RetainedMessageSerializer.deserializeKey(iterator.key())).equals(lastTopic)) {
                    iterator.next();
                }
            }
            int usedMemory = 0;
            ImmutableMap.Builder retrievedMessages = ImmutableMap.builder();
            String lastFoundTopic = null;
            while (iterator.isValid() && usedMemory < maxMemory) {
                String deserializedTopic = RetainedMessageSerializer.deserializeKey(iterator.key());
                RetainedMessage deserializedMessage = RetainedMessageSerializer.deserializeValue(iterator.value());
                if (deserializedMessage.hasExpired()) {
                    iterator.next();
                    continue;
                }
                byte[] payload = this.payloadPersistence.get(deserializedMessage.getPublishId());
                if (payload == null) {
                    log.warn("Could not dereference payload for retained message on topic \"{}\" with payload id \"{}\".", (Object)deserializedTopic, (Object)deserializedMessage.getPublishId());
                    iterator.next();
                    continue;
                }
                deserializedMessage.setMessage(payload);
                lastFoundTopic = deserializedTopic;
                usedMemory += deserializedMessage.getEstimatedSizeInMemory();
                retrievedMessages.put((Object)lastFoundTopic, (Object)deserializedMessage);
                iterator.next();
            }
            BucketChunkResult<ImmutableMap> bucketChunkResult = new BucketChunkResult<ImmutableMap>(retrievedMessages.build(), !iterator.isValid(), lastFoundTopic, bucketIndex);
            return bucketChunkResult;
        }
    }

    @Override
    public void iterate(@NotNull RetainedMessageLocalPersistence.ItemCallback callback) {
        ThreadPreConditions.startsWith("single-writer");
        for (RocksDB bucket : this.buckets) {
            try (RocksIterator iterator = bucket.newIterator();){
                iterator.seekToFirst();
                while (iterator.isValid()) {
                    RetainedMessage message = RetainedMessageSerializer.deserializeValue(iterator.value());
                    String topic = RetainedMessageSerializer.deserializeKey(iterator.key());
                    callback.onItem(topic, message);
                    iterator.next();
                }
            }
        }
    }
}

