/*
 * Decompiled with CFR 0.152.
 */
package com.openshift.internal.restclient;

import com.openshift.internal.restclient.URLBuilder;
import com.openshift.internal.restclient.model.KubernetesResource;
import com.openshift.restclient.IApiTypeMapper;
import com.openshift.restclient.IClient;
import com.openshift.restclient.IOpenShiftWatchListener;
import com.openshift.restclient.IResourceFactory;
import com.openshift.restclient.IWatcher;
import com.openshift.restclient.OpenShiftException;
import com.openshift.restclient.authorization.ResourceForbiddenException;
import com.openshift.restclient.authorization.UnauthorizedException;
import com.openshift.restclient.http.IHttpConstants;
import com.openshift.restclient.model.IList;
import com.openshift.restclient.model.IResource;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.lang.StringUtils;
import org.eclipse.jetty.util.component.LifeCycle;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.eclipse.jetty.websocket.api.Session;
import org.eclipse.jetty.websocket.api.UpgradeException;
import org.eclipse.jetty.websocket.api.WebSocketAdapter;
import org.eclipse.jetty.websocket.client.ClientUpgradeRequest;
import org.eclipse.jetty.websocket.client.WebSocketClient;
import org.jboss.dmr.ModelNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WatchClient
implements IHttpConstants,
IWatcher {
    private static final Logger LOGGER = LoggerFactory.getLogger(WatchClient.class);
    private static final long DEFAULT_LOCK_TIMEOUT = 30000L;
    private URL baseUrl;
    private IApiTypeMapper typeMappings;
    private IResourceFactory factory;
    private IClient client;
    private static WebSocketClient wsClient;
    private static AtomicReference<Status> status;
    private static Lock lock;
    private static Condition isStarted;
    private static long lockTimeout;

    private static WebSocketClient createWebSocketClient() {
        WebSocketClient wsClient = WatchClient.newWebSocketClient();
        wsClient.addLifeCycleListener(new LifeCycle.Listener(){

            public void lifeCycleStopping(LifeCycle event) {
                status.set(Status.Stopping);
            }

            public void lifeCycleStopped(LifeCycle event) {
                status.set(Status.Stopped);
            }

            public void lifeCycleStarting(LifeCycle event) {
                status.set(Status.Starting);
            }

            public void lifeCycleStarted(LifeCycle event) {
                status.set(Status.Started);
                try {
                    if (lock.tryLock(lockTimeout, TimeUnit.MILLISECONDS)) {
                        isStarted.signalAll();
                    }
                }
                catch (InterruptedException e) {
                    LOGGER.debug("Exception while trying to get lock", (Throwable)e);
                }
                finally {
                    lock.unlock();
                }
            }

            public void lifeCycleFailure(LifeCycle event, Throwable cause) {
                LOGGER.error("The watchclient failed:", cause);
                status.set(Status.Stopped);
            }
        });
        return wsClient;
    }

    private static long getLockTimeout() {
        try {
            return Long.parseLong(System.getProperty("com.openshift.restclient.watchlocktimeoutms", String.valueOf(30000L)));
        }
        catch (NumberFormatException e) {
            return 30000L;
        }
    }

    public WatchClient(URL baseUrl, IApiTypeMapper typeMappings, IClient client) {
        this.baseUrl = baseUrl;
        this.typeMappings = typeMappings;
        this.factory = client.getResourceFactory();
        this.client = client;
    }

    public IWatcher watch(Collection<String> kinds, String namespace, IOpenShiftWatchListener listener) {
        try {
            ClientUpgradeRequest request = this.newRequest(this.client.getAuthorizationStrategy().getToken());
            for (String kind : kinds) {
                WatchEndpoint socket = new WatchEndpoint(listener, kind);
                String resourceVersion = this.getResourceVersion(kind, namespace, socket);
                String endpoint = new URLBuilder(this.baseUrl, this.typeMappings).kind(kind).namespace(namespace).watch().addParmeter("resourceVersion", resourceVersion).websocket();
                this.connect(socket, endpoint, request);
            }
        }
        catch (Exception e) {
            throw this.createOpenShiftException(String.format("Could not watch resources in namespace %s: %s", namespace, e.getMessage()), e);
        }
        return this;
    }

    private void connect(WatchEndpoint socket, String endpoint, ClientUpgradeRequest request) throws Exception {
        this.start();
        if (status.get() == Status.Starting) {
            isStarted.await(lockTimeout, TimeUnit.MILLISECONDS);
        }
        wsClient.connect((Object)socket, new URI(endpoint), request);
    }

    public void start() {
        if (status.get() == Status.Started || status.get() == Status.Starting) {
            return;
        }
        try {
            wsClient.start();
        }
        catch (Exception e) {
            throw this.createOpenShiftException(String.format("Could not start watchClient", new Object[0]), e);
        }
    }

    @Override
    public void stop() {
        if (status.get() == Status.Stopping || status.get() == Status.Stopped) {
            return;
        }
        try {
            wsClient.stop();
        }
        catch (Exception e) {
            LOGGER.debug("Unable to stop the watch client", (Throwable)e);
        }
    }

    private ClientUpgradeRequest newRequest(String token) {
        ClientUpgradeRequest request = new ClientUpgradeRequest();
        request.setHeader("Origin", this.baseUrl.toString());
        request.setHeader("User-Agent", "openshift-restclient-java");
        request.setHeader("Authorization", "Bearer " + token);
        return request;
    }

    private static WebSocketClient newWebSocketClient() {
        SslContextFactory factory = new SslContextFactory();
        factory.setTrustAll(true);
        WebSocketClient client = new WebSocketClient(factory);
        return client;
    }

    private String getResourceVersion(String kind, String namespace, WatchEndpoint endpoint) throws Exception {
        IList list = this.client.get(kind, namespace);
        Collection<IResource> items = list.getItems();
        ArrayList<IResource> resources = new ArrayList<IResource>(items.size());
        resources.addAll(items);
        endpoint.setResources(resources);
        return list.getMetadata().get("resourceVersion");
    }

    private OpenShiftException createOpenShiftException(String message, Throwable e) {
        LOGGER.debug(message, e);
        int responseCode = 0;
        if (e instanceof UpgradeException) {
            UpgradeException ex = (UpgradeException)e;
            responseCode = ex.getResponseStatusCode();
        }
        switch (responseCode) {
            case 403: {
                return new ResourceForbiddenException("Resource Forbidden", e);
            }
            case 401: {
                return new UnauthorizedException(this.client.getAuthorizationDetails(this.baseUrl.toString()));
            }
        }
        return new OpenShiftException(e, message, new Object[0]);
    }

    static {
        status = new AtomicReference<Status>(Status.Stopped);
        lock = new ReentrantLock();
        isStarted = lock.newCondition();
        lockTimeout = WatchClient.getLockTimeout();
        wsClient = WatchClient.createWebSocketClient();
    }

    private class WatchEndpoint
    extends WebSocketAdapter {
        private IOpenShiftWatchListener listener;
        private List<IResource> resources;
        private final String kind;

        public WatchEndpoint(IOpenShiftWatchListener listener, String kind) {
            this.listener = listener;
            this.kind = kind;
        }

        public void setResources(List<IResource> resources) {
            this.resources = resources;
        }

        public void onWebSocketClose(int statusCode, String reason) {
            LOGGER.debug("WatchSocket closed for kind {}", (Object)this.kind);
            this.getSession().close(statusCode, reason);
            super.onWebSocketClose(statusCode, reason);
            this.listener.disconnected();
        }

        public void onWebSocketConnect(Session session) {
            LOGGER.debug("WatchSocket connected {}", (Object)this.kind);
            super.onWebSocketConnect(session);
            this.listener.connected(this.resources);
        }

        public void onWebSocketError(Throwable err) {
            LOGGER.debug("WatchSocket Error for kind " + this.kind, err);
            this.listener.error(WatchClient.this.createOpenShiftException("WatchSocket Error", err));
        }

        public void onWebSocketText(String message) {
            LOGGER.debug(message);
            KubernetesResource payload = (KubernetesResource)WatchClient.this.factory.create(message);
            ModelNode node = payload.getNode();
            IOpenShiftWatchListener.ChangeType event = new IOpenShiftWatchListener.ChangeType(node.get("type").asString());
            Object resource = WatchClient.this.factory.create(node.get("object").toJSONString(true));
            if (StringUtils.isEmpty((String)resource.getKind())) {
                LOGGER.error("Unable to determine resource kind from: " + node.get("object").toJSONString(false));
            }
            this.listener.received((IResource)resource, event);
        }
    }

    private static enum Status {
        Started,
        Starting,
        Stopped,
        Stopping;

    }
}

