package io.a2a.spec;

import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.a2a.util.Assert;

/**
 * Represents a file, data structure, or other resource generated by an agent during a task.
 */
@JsonInclude(JsonInclude.Include.NON_ABSENT)
@JsonIgnoreProperties(ignoreUnknown = true)
public record Artifact(String artifactId, String name, String description, List<Part<?>> parts, Map<String, Object> metadata) {

    public Artifact {
        Assert.checkNotNullParam("artifactId", artifactId);
        Assert.checkNotNullParam("parts", parts);
        if (parts.isEmpty()) {
            throw new IllegalArgumentException("Parts cannot be empty");
        }
    }

    public static class Builder {
        private String artifactId;
        private String name;
        private String description;
        private List<Part<?>> parts;
        private Map<String, Object> metadata;

        public Builder(){
        }

        public Builder(Artifact existingArtifact) {
            artifactId = existingArtifact.artifactId;
            name = existingArtifact.name;
            description = existingArtifact.description;
            parts = existingArtifact.parts;
            metadata = existingArtifact.metadata;
        }

        public Builder artifactId(String artifactId) {
            this.artifactId = artifactId;
            return this;
        }


        public Builder name(String name) {
            this.name = name;
            return this;
        }

        public Builder description(String description) {
            this.description = description;
            return this;
        }

        public Builder parts(List<Part<?>> parts) {
            this.parts = parts;
            return this;
        }

        public Builder parts(Part<?>... parts) {
            this.parts = List.of(parts);
            return this;
        }

        public Builder metadata(Map<String, Object> metadata) {
            this.metadata = metadata;
            return this;
        }

        public Artifact build() {
            return new Artifact(artifactId, name, description, parts, metadata);
        }
    }
}
