/*
 * JBoss, Home of Professional Open Source
 * Copyright 2007, Red Hat Middleware LLC, and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.metadata.annotation.creator.ejb;

import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Method;

import javax.ejb.TransactionAttribute;

import org.jboss.metadata.annotation.creator.AbstractFinderUser;
import org.jboss.metadata.annotation.creator.MethodParametersHelper;
import org.jboss.metadata.annotation.finder.AnnotationFinder;
import org.jboss.metadata.common.ejb.IEjbJarMetaData;
import org.jboss.metadata.ejb.spec.AssemblyDescriptorMetaData;
import org.jboss.metadata.ejb.spec.ContainerTransactionMetaData;
import org.jboss.metadata.ejb.spec.ContainerTransactionsMetaData;
import org.jboss.metadata.ejb.spec.EjbJarMetaData;
import org.jboss.metadata.ejb.spec.EnterpriseBeanMetaData;
import org.jboss.metadata.ejb.spec.MethodMetaData;
import org.jboss.metadata.ejb.spec.MethodParametersMetaData;
import org.jboss.metadata.ejb.spec.MethodsMetaData;
import org.jboss.metadata.ejb.spec.TransAttributeType;

/**
 * E defines the actual annotated element we're working for.
 *
 * @author <a href="mailto:carlo.dewolf@jboss.com">Carlo de Wolf</a>
 * @version $Revision: 67165 $
 */
public abstract class AbstractTransactionAttributeProcessor<E extends AnnotatedElement, T extends EnterpriseBeanMetaData> extends AbstractFinderUser
{
   protected AbstractTransactionAttributeProcessor(AnnotationFinder<AnnotatedElement> finder)
   {
      super(finder);
   }
   
   protected abstract ContainerTransactionMetaData createContainerTransaction(String ejbName, TransactionAttribute annotation, E element);
   
   private MethodMetaData createMethod(String ejbName, Method method)
   {
      MethodMetaData methodMetaData = new MethodMetaData();
      methodMetaData.setEjbName(ejbName);
      if(method == null)
         methodMetaData.setMethodName("*");
      else
      {
         methodMetaData.setMethodName(method.getName());
         MethodParametersMetaData methodParameters = MethodParametersHelper.create(method);
         if(methodParameters != null)
            methodMetaData.setMethodParams(methodParameters);
      }
      return methodMetaData;
   }
   
   protected MethodsMetaData createMethods(String ejbName, Method method)
   {
      MethodsMetaData methods = new MethodsMetaData();
      methods.add(createMethod(ejbName, method));
      return methods;
   }
   
   protected TransAttributeType createTransAttributeType(TransactionAttribute annotation)
   {
      switch(annotation.value())
      {
         case MANDATORY:
            return TransAttributeType.Mandatory;
         case NEVER:
            return TransAttributeType.Never;
         case NOT_SUPPORTED:
            return TransAttributeType.NotSupported;
         case REQUIRED:
            return TransAttributeType.Required;
         case REQUIRES_NEW:
            return TransAttributeType.RequiresNew;
         case SUPPORTS:
            return TransAttributeType.Supports;
      }
      throw new IllegalArgumentException("Unknown transaction attribute value " + annotation.value());
   }

   public void process(T bean, E element)
   {
      TransactionAttribute annotation = finder.getAnnotation(element, TransactionAttribute.class);
      if(annotation == null)
         return;
      
      IEjbJarMetaData ejbJarMetaData = bean.getEjbJarMetaData();
      
      if(ejbJarMetaData.getAssemblyDescriptor() == null)
         ejbJarMetaData.setAssemblyDescriptor(new AssemblyDescriptorMetaData());
      if(ejbJarMetaData.getAssemblyDescriptor().getContainerTransactions() == null)
         ejbJarMetaData.getAssemblyDescriptor().setContainerTransactions(new ContainerTransactionsMetaData());
      
      ContainerTransactionMetaData transaction = createContainerTransaction(bean.getEjbName(), annotation, element);
      ejbJarMetaData.getAssemblyDescriptor().getContainerTransactions().add(transaction);
   }
}
