/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */ 
package org.jboss.security.plugins;

import java.security.Principal;
import java.security.acl.Group; 
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map; 
 
import org.jboss.logging.Logger;
import org.jboss.security.AuthenticationManager;
import org.jboss.security.AuthorizationManager;
import org.jboss.security.RunAs;
import org.jboss.security.SecurityConstants;
import org.jboss.security.SecurityContext;  
import org.jboss.security.SecurityContextUtil;
import org.jboss.security.SecurityUtil;
import org.jboss.security.SubjectInfo; 
import org.jboss.security.audit.AuditManager;
import org.jboss.security.identitytrust.IdentityTrustManager;
import org.jboss.security.mapping.MappingManager; 
import org.jboss.security.plugins.audit.JBossAuditManager;
import org.jboss.security.plugins.identitytrust.JBossIdentityTrustManager;
import org.jboss.security.plugins.mapping.JBossMappingManager;

import static org.jboss.security.SecurityConstants.ROLES_IDENTIFIER;

/**
 *  Implementation of the Security Context for the JBoss AS
 *  @author <a href="mailto:Anil.Saldhana@jboss.org">Anil Saldhana</a>
 *  @version $Revision$
 *  @since  Aug 30, 2006
 */
public class JBossSecurityContext implements SecurityContext
{  
   private static final long serialVersionUID = 1L;
   protected static final Logger log = Logger.getLogger(JBossSecurityContext.class); 
   protected boolean trace = log.isTraceEnabled();  
   
   protected Map<String,Object> contextData = new HashMap<String,Object>();
   
   protected String securityDomain = SecurityConstants.DEFAULT_APPLICATION_POLICY;

   protected SubjectInfo subjectInfo = null;
    
   protected RunAs incomingRunAs = null;
   protected RunAs outgoingRunAs = null;
   
   public JBossSecurityContext(String securityDomain)
   {
      this.securityDomain = securityDomain;
   }

   /**
    * @see SecurityContext#getAuthenticationManager()
    */
   public AuthenticationManager getAuthenticationManager()
   {
      return SecurityUtil.getAuthenticationManager(securityDomain);
   }

   /**
    * @see SecurityContext#getAuthorizationManager()
    */
   public AuthorizationManager getAuthorizationManager()
   { 
      return SecurityUtil.getAuthorizationManager(securityDomain);
   }

   /**
    * @see SecurityContext#getData()
    */ 
   public Map<String,Object> getData()
   { 
      return contextData;
   }
 
   public String getSecurityDomain()
   { 
      return securityDomain;
   }
 

   /**
    * @see SecurityContext#getSubjectInfo()
    */
   public SubjectInfo getSubjectInfo()
   { 
      return subjectInfo;
   } 
   
   /**
    * @see SecurityContext#getAuditContext()
    */
   public AuditManager getAuditManager()
   {
      return new JBossAuditManager(securityDomain);
   }
   
   /**
    * @see SecurityContext#getMappingManager()
    */
   public MappingManager getMappingManager()
   {
      return new JBossMappingManager(this.securityDomain);
   } 

   /**
    * @see SecurityContext#getIdentityTrustManager()
    */
   public IdentityTrustManager getIdentityTrustManager()
   { 
      return new JBossIdentityTrustManager(this);
   } 
   
   /**
    * @see SecurityContext#getOutgoingRunAs()
    */
   public RunAs getIncomingRunAs()
   { 
      return this.incomingRunAs;
   }

   /**
    * @see SecurityContext#setOutgoingRunAs(RunAs)
    */
   public void setIncomingRunAs(RunAs runAs)
   { 
      this.incomingRunAs = runAs;
   } 

   /**
    * @see SecurityContext#getOutgoingRunAs()
    */
   public RunAs getOutgoingRunAs()
   { 
      return this.outgoingRunAs;
   }

   /**
    * @see SecurityContext#setOutgoingRunAs(RunAs)
    */
   public void setOutgoingRunAs(RunAs runAs)
   { 
      this.outgoingRunAs = runAs;
   } 
   
   /**
    * @see SecurityContext#getUtil()
    */
   public SecurityContextUtil getUtil()
   {  
      return SecurityContextFactory.createUtil(this);
   } 
   
   //Value Added Methods
   
   public void setSubjectInfo(SubjectInfo si)
   {
      this.subjectInfo = si;
   }
   
   public void setRoles(Group roles, boolean replace)
   {
      Group mergedRoles = roles;
      if(!replace)
      {
         mergedRoles = mergeGroups( (Group)contextData.get(ROLES_IDENTIFIER), roles); 
      } 
      contextData.put(ROLES_IDENTIFIER, mergedRoles);
   }
   
   private Group mergeGroups(Group a, Group b)
   {
      Group newGroup = b;
      if(a != null)
      {
         Enumeration en = a.members();
         while(en.hasMoreElements())
         {
            newGroup.addMember((Principal)en.nextElement());
         } 
      } 
      return newGroup; 
   } 
   

   @Override
   public Object clone() throws CloneNotSupportedException
   { 
      JBossSecurityContext jsc = (JBossSecurityContext) super.clone();
      if(jsc != null)
         jsc.contextData = (Map<String, Object>) ((HashMap)contextData).clone();
      return super.clone();
   }

} 
