/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.security.authorization.modules.ejb;

import java.lang.reflect.Method;
import java.security.Principal;
import java.util.Map;

import javax.security.jacc.PolicyContext; 
import org.jboss.logging.Logger; 
import org.jboss.security.AuthorizationManager;
import org.jboss.security.authorization.AuthorizationContext;
import org.jboss.security.authorization.PolicyRegistration;
import org.jboss.security.authorization.Resource; 
import org.jboss.security.authorization.ResourceKeys;
import org.jboss.security.authorization.modules.AuthorizationModuleDelegate;
import org.jboss.security.authorization.sunxacml.JBossXACMLUtil; 

import com.sun.xacml.Policy;
import com.sun.xacml.ctx.RequestCtx;

//$Id: EJBXACMLPolicyModuleDelegate.java 62910 2007-05-08 21:52:04Z anil.saldhana@jboss.com $

/**
 *  Authorization Module Delegate that deals with the authorization decisions
 *  for the EJB Layer
 *  @author <a href="mailto:Anil.Saldhana@jboss.org">Anil Saldhana</a>
 *  @since  Jul 6, 2006 
 *  @version $Revision: 62910 $
 */
public class EJBXACMLPolicyModuleDelegate extends AuthorizationModuleDelegate
{   
   private String ejbName = null;
   private Method ejbMethod = null;
   private Principal principal = null; 
   
   public EJBXACMLPolicyModuleDelegate()
   {
     log = Logger.getLogger(getClass());
     trace = log.isTraceEnabled();
   }
   
   /**
    * @see AuthorizationModuleDelegate#authorize(Resource)
    */
   public int authorize(Resource resource)
   {
      //Get the contextual map
      Map map = resource.getMap();
      if(map == null)
         throw new IllegalStateException("Map from the Resource is null");
    
      if(map.size() == 0)
         throw new IllegalStateException("Map from the Resource is size zero");
      AuthorizationManager am = (AuthorizationManager) map.get("authorizationManager");
      if(am == null)
         throw new IllegalStateException("Authorization Manager is null");
      if(am instanceof PolicyRegistration)
         this.policyRegistration = (PolicyRegistration) am; 
      //Populate local variables from the resource
      this.ejbName = (String)map.get(ResourceKeys.EJB_NAME);
      this.ejbMethod = (Method)map.get(ResourceKeys.EJB_METHOD); 
      this.principal = (Principal)map.get(ResourceKeys.EJB_PRINCIPAL);
      Boolean roleRefCheck = checkBooleanValue((Boolean)map.get(ResourceKeys.ROLEREF_PERM_CHECK)); 
      if(roleRefCheck)
         throw new IllegalStateException("SECURIY-50:Role Ref checks not implemented");
      
      return process(am);
   } 
   
   //Private Methods
   /**
    * Process the web request
    * @param request
    * @param sc
    * @return
    */
   private int process(AuthorizationManager am) 
   { 
      int result = AuthorizationContext.DENY;
      EJBXACMLUtil util = new EJBXACMLUtil();
      try
      {
         RequestCtx requestCtx = util.createXACMLRequest(this.ejbName,
               this.ejbMethod.getName(),this.principal, 
               am.getUserRoles(principal));
         String contextID = PolicyContext.getContextID();
         if(contextID == null)
            throw new IllegalStateException("Context ID is null");
         Policy policy = (Policy)policyRegistration.getPolicy(contextID,null);
         if(policy == null)
         {
            if(trace)
               log.trace("Policy obtained is null for contextID:"+contextID);
            throw new IllegalStateException("Missing xacml policy for contextid:"+contextID);
         }
         result = JBossXACMLUtil.checkXACMLAuthorization(requestCtx,policy);
      }
      catch(Exception e)
      {
         if(trace)
            log.trace("Exception in processing:",e);
         result = AuthorizationContext.DENY;
      }  
      return result;
   } 
   
   /**
    * Ensure that the bool is a valid value
    * @param bool
    * @return bool or Boolean.FALSE (when bool is null)
    */
   private Boolean checkBooleanValue(Boolean bool)
   {
      if(bool == null)
         return Boolean.FALSE;
      return bool;
   } 
}
