/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.security.plugins;
 
import java.io.InputStream;
import java.net.URL;
import java.security.Principal;
import java.security.PrivilegedActionException;
import java.security.acl.Group;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
 
import javax.security.auth.Subject;
import javax.security.auth.callback.CallbackHandler;
import javax.security.jacc.PolicyContext;
import javax.security.jacc.PolicyContextException;

import org.jboss.logging.Logger;
import org.jboss.security.AnybodyPrincipal;
import org.jboss.security.AuthorizationManager;
import org.jboss.security.NobodyPrincipal;  
import org.jboss.security.SecurityConstants;
import org.jboss.security.SecurityContext;
import org.jboss.security.SecurityRolesAssociation;
import org.jboss.security.SimpleGroup; 
import org.jboss.security.authorization.AuthorizationContext;
import org.jboss.security.authorization.AuthorizationException;
import org.jboss.security.authorization.PolicyRegistration;
import org.jboss.security.authorization.Resource;
import org.jboss.security.mapping.MappingContext; 
import org.jboss.security.mapping.MappingManager;
import org.jboss.security.plugins.authorization.JBossAuthorizationContext;
import org.jboss.util.xml.DOMUtils;
import org.w3c.dom.Element;

import com.sun.xacml.Policy;

import static org.jboss.security.SecurityConstants.ROLES_IDENTIFIER;

//$Id: JBossAuthorizationManager.java 65313 2007-09-11 22:37:23Z anil.saldhana@jboss.com $

/**
 *  Authorization Manager implementation
 *  @author <a href="mailto:Anil.Saldhana@jboss.org">Anil Saldhana</a>
 *  @since  Jan 3, 2006 
 *  @version $Revision: 65313 $
 */
public class JBossAuthorizationManager 
implements AuthorizationManager,PolicyRegistration
{  
   private String securityDomain; 
   
   private Map contextIdToPolicy = new HashMap(); 
   
   private static Logger log = Logger.getLogger(JBossAuthorizationManager.class);
   
   protected boolean trace = log.isTraceEnabled();

   private CallbackHandler callbackHandler = null;
   
   private AuthorizationContext authorizationContext = null;
   
   public JBossAuthorizationManager(String securityDomainName)
   {
      this.securityDomain = securityDomainName;
   }
   
   public JBossAuthorizationManager(String securityDomainName, CallbackHandler cbh)
   {
      this(securityDomainName);
      this.callbackHandler = cbh;
   }
   
   /**
    * @see AuthorizationManager#authorize(Resource)
    */
   public int authorize(Resource resource) throws AuthorizationException
   {
      String SUBJECT_CONTEXT_KEY = SecurityConstants.SUBJECT_CONTEXT_KEY;
      Subject subject = null;
      try
      {
         subject = (Subject) PolicyContext.getContext(SUBJECT_CONTEXT_KEY);
      }
      catch (PolicyContextException e)
      {
         log.error("Error obtaining AuthenticatedSubject:",e);
      }
      if(this.authorizationContext == null)
         this.authorizationContext = new JBossAuthorizationContext(this.securityDomain,subject,
                                          this.callbackHandler ); 
      return this.authorizationContext.authorize(resource);
   }  
   
   /** Does the current Subject have a role(a Principal) that equates to one
    of the role names. This method obtains the Group named 'Roles' from
    the principal set of the currently authenticated Subject as determined
    by the SecurityAssociation.getSubject() method and then creates a
    SimplePrincipal for each name in roleNames. If the role is a member of the
    Roles group, then the user has the role. This requires that the caller
    establish the correct SecurityAssociation subject prior to calling this
    method. In the past this was done as a side-effect of an isValid() call,
    but this is no longer the case.
    
    @param principal - ignored. The current authenticated Subject determines
    the active user and assigned user roles.
    @param rolePrincipals - a Set of Principals for the roles to check.
    
    @see java.security.acl.Group;
    @see Subject#getPrincipals()
    */
   public boolean doesUserHaveRole(Principal principal, Set rolePrincipals)
   {
      boolean hasRole = false;
      Group roles = this.getCurrentRoles(principal);
      if( trace )
         log.trace("doesUserHaveRole(Set), roles: "+roles);
      if(roles != null)
      {
         Iterator iter = rolePrincipals.iterator();
         while( hasRole == false && iter.hasNext() )
         {
            Principal role = (Principal) iter.next();
            hasRole = doesRoleGroupHaveRole(role, roles);
            if( trace )
               log.trace("hasRole("+role+")="+hasRole);
         }
         if( trace )
            log.trace("hasRole="+hasRole);
      } 
      return hasRole;
   }
   
   /** Does the current Subject have a role(a Principal) that equates to one
    of the role names.
    
    @see #doesUserHaveRole(Principal, Set)
    
    @param principal - ignored. The current authenticated Subject determines
    the active user and assigned user roles.
    @param role - the application domain role that the principal is to be
    validated against.
    @return true if the active principal has the role, false otherwise.
    */
   public boolean doesUserHaveRole(Principal principal, Principal role)
   {
      boolean hasRole = false;
      Group roles = this.getCurrentRoles(principal);
      hasRole = doesRoleGroupHaveRole(role, roles); 
      return hasRole;
   } 
   
   /** Return the set of domain roles the current active Subject 'Roles' group
    found in the subject Principals set.
    
    @param principal - ignored. The current authenticated Subject determines
    the active user and assigned user roles.
    @return The Set<Principal> for the application domain roles that the
    principal has been assigned.
    */
   public Set getUserRoles(Principal principal)
   { 
      Group userRoles = getCurrentRoles(principal);
      return this.getRolesAsSet(userRoles); 
   }  
     
   
   /** Check that the indicated application domain role is a member of the
    user's assigned roles. This handles the special AnybodyPrincipal and
    NobodyPrincipal independent of the Group implementation.
    
    @param role , the application domain role required for access
    @param userRoles , the set of roles assigned to the user
    @return true if role is in userRoles or an AnybodyPrincipal instance, false
    if role is a NobodyPrincipal or no a member of userRoles
    */
   protected boolean doesRoleGroupHaveRole(Principal role, Group userRoles)
   {
      // First check that role is not a NobodyPrincipal
      if (role instanceof NobodyPrincipal)
         return false;
      
      // Check for inclusion in the user's role set
      boolean isMember = userRoles.isMember(role);
      if (isMember == false)
      {   // Check the AnybodyPrincipal special cases
         isMember = (role instanceof AnybodyPrincipal);
      }
      
      return isMember;
   } 

   /**
    * @see PolicyRegistration#registerPolicy(String, URL)
    */
   public void registerPolicy(String contextID, URL location) 
   {
      try
      { 
         if(trace)
            log.trace("Registering policy for contextId:" +
                         contextID + " and location:"+location.getPath()); 
         registerPolicy( contextID, location.openStream()); 
      }
      catch(Exception e)
      {
         log.debug("Error in registering xacml policy:",e);
      }  
   }
   
   /**
    * @see PolicyRegistration#registerPolicy(String, InputStream)
    */
   public void registerPolicy(String contextID, InputStream stream) 
   {
      try
      {  
         Element elm = DOMUtils.parse(stream);
         Policy policy = Policy.getInstance(elm); 
         this.contextIdToPolicy.put(contextID, policy);
      }
      catch(Exception e)
      {
         log.debug("Error in registering xacml policy:",e);
      }  
   }

   /**
    * @see PolicyRegistration#deRegisterPolicy(String)
    */
   public void deRegisterPolicy(String contextID)
   { 
      this.contextIdToPolicy.remove(contextID);
      if(trace)
         log.trace("DeRegistered policy for contextId:" + contextID);
   }

   /**
    * @see PolicyRegistration#getPolicy(String, Map)
    */
   public Object getPolicy(String contextID, Map contextMap)
   {
      return this.contextIdToPolicy.get(contextID);
   }

   /**
    * 
    */
   public String toString()
   {
      StringBuffer buf = new StringBuffer();
      buf.append("[AuthorizationManager:class=").append(getClass().getName());
      buf.append(":").append(this.securityDomain).append(":");
      buf.append("]");
      return buf.toString();
   } 
   
   //Value added methods
   /**
    * Set the AuthorizationContext
    */
   public void setAuthorizationContext(AuthorizationContext ac)
   {
      if(ac == null)
         throw new IllegalArgumentException("AuthorizationContext is null");
      this.authorizationContext = ac;
   }
   
   public String getSecurityDomain()
   {
      return this.securityDomain;
   }  

   //Private Methods
   private HashSet getRolesAsSet(Group roles)
   {
      HashSet userRoles = null;
      if( roles != null )
      {
         userRoles = new HashSet();
         Enumeration members = roles.members();
         while( members.hasMoreElements() )
         {
            Principal role = (Principal) members.nextElement();
            userRoles.add(role);
         }
      }
      return userRoles;
   } 
   
   /*
    * Get the current role group from the security context or
    * the Subject
    * @param principal The Principal in question
    */
   private Group getCurrentRoles(Principal principal)
   {
      boolean emptyContextRoles = false;
      //Check that the caller is authenticated to the current thread
      Subject subject = null;
      try
      {
         subject = SubjectActions.getActiveSubject();
      }
      catch (PrivilegedActionException e)
      {
         throw new IllegalStateException(e);
      } 
      Group subjectRoles = getSubjectRoles(subject);
      
      //Deal with the security context
      SecurityContext sc = SubjectActions.getSecurityContext(); 
      if(sc == null)
      {
         sc = new JBossSecurityContext(securityDomain); 
         SubjectActions.setSecurityContext(sc);   
      } 

      Group userRoles = (Group)sc.getData().get(ROLES_IDENTIFIER);
      if(userRoles == null || "true".equalsIgnoreCase(SubjectActions.getRefreshSecurityContextRoles()))
         emptyContextRoles = true;
      userRoles = copyGroups(userRoles, subjectRoles); 
      
      /**
       * Update the roles in the SecurityContext and
       * allow mapping rules be applied only if the SC roles
       * and the subject roles are not the same
       */
      if(subjectRoles != userRoles || emptyContextRoles)
      { 
         MappingManager mm = sc.getMappingManager();
         MappingContext mc = mm.getMappingContext(Group.class);
         if(mc != null)
         {
            Map contextMap = new HashMap();
            contextMap.put(SecurityConstants.ROLES_IDENTIFIER, userRoles);
            contextMap.put(SecurityConstants.PRINCIPAL_IDENTIFIER, principal);
            //Append any deployment role->principals configuration done by the user
            contextMap.put(SecurityConstants.DEPLOYMENT_PRINCIPAL_ROLES_MAP,
                  SecurityRolesAssociation.getSecurityRoles());
            
            //Append the principals also
            contextMap.put(SecurityConstants.PRINCIPALS_SET_IDENTIFIER, subject.getPrincipals());
            if(trace)
               log.trace("Roles before mapping:"+ userRoles);
            mc.performMapping(contextMap, userRoles);
            if(trace)
               log.trace("Roles after mapping:"+ userRoles);
         } 
         sc.getData().put(ROLES_IDENTIFIER, userRoles); 
      } 

      //Send the final processed (mapping applied) roles
      return userRoles;
   } 
   
   /**
    * Copy the principals from the second group into the first.
    * If the first group is null and the second group is not, the
    * first group will be made equal to the second group
    * @param source
    * @param toCopy
    */
   private Group copyGroups(Group source, Group toCopy)
   {
      if(toCopy == null)
         return source;
      if(source == null && toCopy != null) 
         source = new SimpleGroup(SecurityConstants.ROLES_IDENTIFIER);
      Enumeration en = toCopy.members();
      while(en.hasMoreElements())
      {
         source.addMember((Principal)en.nextElement());
      }
       
      return source;
   }

   /**
    * @see AuthorizationManager#getTargetRoles(Principal, Map)
    */
   public Group getTargetRoles(Principal targetPrincipal, Map contextMap)
   {
      throw new RuntimeException("Not implemented");
   }
   
   /**
    * Get the Subject roles by looking for a Group called 'Roles'
    * @param theSubject - the Subject to search for roles
    * @return the Group contain the subject roles if found, null otherwise
    */
   private Group getSubjectRoles(Subject theSubject)
   {
      if(theSubject == null)
         throw new IllegalArgumentException("Subject is null");
      Set subjectGroups = theSubject.getPrincipals(Group.class);
      Iterator iter = subjectGroups.iterator();
      Group roles = null;
      while( iter.hasNext() )
      {
         Group grp = (Group) iter.next();
         String name = grp.getName();
         if( name.equals(ROLES_IDENTIFIER) )
            roles = grp;
      }
      return roles;
   }
}
