package org.jboss.maven.plugins.retro;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;
import java.util.zip.Deflater;

import org.apache.commons.io.FileUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.project.MavenProject;
import org.jboss.ant.tasks.retro.Weaver;

/**
 * Maven wrapper for JBoss Retro.  By default
 * it will bind to the process classes phase, which takes
 * place immediately after the compile phase.
 * 
 * @phase process-classes
 * @goal retro-compile
 * 
 */
public class RetroMojo extends AbstractMojo
{

   //private static String lSep = System.getProperty("line.separator");

   /**
    * The Maven Project Object
    *
    * @parameter expression="${project}"
    * @required
    */
   protected MavenProject project;

   /**
    * The Maven Project Helper Object
    *
    * @component
    * @required
    */
   protected org.apache.maven.project.MavenProjectHelper projectHelper;

   /**
    * The Maven Plugin Object
    *
    * @parameter expression="${plugin.components}"
    * @required
    * @readonly
    */
   protected List pluginComponents;

   /**
    * The plugin dependencies.
    *
    * @parameter expression="${plugin.artifacts}"
    * @required
    * @readonly
    */
   private List pluginArtifacts;

   /**
    * Project classpath.
    *
    * @parameter expression="${project.compileClasspathElements}"
    * @required
    * @readonly
    */
   private List classpathElements;

   /**
    * @parameter expression="${project.pluginArtifacts}"
    * @required
    * @readonly
    */
   //private HashSet pluginArtifacts;
   /**
    * The directory for compiled classes.
    *
    * @parameter expression="${project.build.outputDirectory}"
    * @required
    * @readonly
    */
   private File targetClassesDirectory;

   /**
    * The directory for compiled classes.
    *
    * @parameter expression="${project.build.directory}"
    * @required
    * @readonly
    */
   private File targetDirectory;

   /**
    * @parameter
    */
   private boolean verbose = false;

   /**
    * @parameter
    */
   private boolean suppress = true;

   /**
    * List of the jar file entries
    */
   private ArrayList<JarFileEntry> fileEntries = new ArrayList<JarFileEntry>();

   public void execute()
   {
      this.getLog().info("[retro] Compiling classes for jdk14");

      if (!project.getArtifact().getType().equalsIgnoreCase("jar")) {
         this.getLog().info("[retro] Project " + project.getName() + " is not"
               + " a jar project.  No retro compile needed.");
         return;
      }
      ArrayList<String> argsList = new ArrayList<String>();

      if (verbose)
      {
         argsList.add("-verbose");
      }
      if (suppress)
      {
         argsList.add("-suppress");
      }
      argsList.add("-destdir");
      try
      {
         argsList.add(targetDirectory.getCanonicalPath() + "/classes-retro");
      }
      catch (Exception e)
      {
         this.getLog().error(e);
      }
      argsList.add("-cp");
      StringBuilder classpath = new StringBuilder();
      String pathSeparator = System.getProperty("path.separator");
      for (Object element : classpathElements)
      {
         classpath.append(element);
         classpath.append(pathSeparator);
      }

      for (Object artifact : pluginArtifacts)
      {
         try
         {
            File artifactFile = ((Artifact) artifact).getFile();
            if (artifactFile != null)
            {
               classpath.append(artifactFile.getCanonicalPath());
               classpath.append(pathSeparator);
            }
         }
         catch (IOException ioe)
         {
            this.getLog().warn("Could not get filename");
         }
      }

      argsList.add(classpath.toString());

      try
      {
         argsList.add(targetClassesDirectory.getCanonicalPath());
      }
      catch (IOException ioe)
      {
         this.getLog().error(ioe.toString());
      }

      String[] args = new String[argsList.size()];
      for (int i = 0; i < args.length; ++i)
      {
         args[i] = argsList.get(i);
      }

      Weaver weaver = new Weaver();
      try
      {
         weaver.weave(args);
      }
      catch (Exception e)
      {
         this.getLog().error(e);
      }

      // Create jar file and attach it to maven.
      File retroJarFile = new File(targetDirectory.getAbsolutePath() + "/" + project.getArtifactId() + "-jdk14.jar");
      createJarEntries();
      createRetroJarFile(retroJarFile);
      projectHelper.attachArtifact(project, retroJarFile, "jdk14");
   }

   private void createJarEntries()
   {
      try
      {
         File retroClassesDir = new File(targetDirectory.getAbsolutePath() + "/classes-retro");
         Collection retroClasses = FileUtils.listFiles(retroClassesDir, null, true);
         for (Object classFile : retroClasses)
         {
            String relativePath = classFile.toString().replace(retroClassesDir.getAbsolutePath() + "/", "");
            byte[] content = FileUtils.readFileToByteArray((File) classFile);
            fileEntries.add(new JarFileEntry(relativePath, content));
         }
      }
      catch (IOException ioe)
      {
         this.getLog().error("error reading class file: " + ioe);
      }
   }

   private void createRetroJarFile(File retroJarFile)
   {
      try
      {
         Manifest manifest = new Manifest();
         Attributes attributes = manifest.getMainAttributes();
         attributes.putValue("Manifest-Version", "1.0");
         attributes.putValue("Created-By", System.getProperty("java.vm.version") + " ("
               + System.getProperty("java.vm.vendor") + ")");
         JarOutputStream stream = new JarOutputStream(new FileOutputStream(retroJarFile), manifest);
         stream.setLevel(Deflater.BEST_COMPRESSION);
         for (JarFileEntry fileEntry : fileEntries)
         {
            JarEntry jarEntry = new JarEntry(fileEntry.getName());
            stream.putNextEntry(jarEntry);
            stream.write(fileEntry.getContent());
         }
         stream.close();
      }
      catch (IOException ioe)
      {
         this.getLog().error("Unable to write retro jar: " + ioe);
      }
   }

   private static class JarFileEntry
   {

      private byte[] content;

      private String name;

      public JarFileEntry(String name, byte[] content)
      {
         this.name = name;
         this.content = content;
      }

      public byte[] getContent()
      {
         return content;
      }

      public String getName()
      {
         return name;
      }

      public void setName(String name)
      {
         this.name = name;
      }
   }
}
