/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.RowMutation;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.ReplayPosition;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.PureJavaCrc32;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLogSegment {
    private static final Logger logger = LoggerFactory.getLogger(CommitLogSegment.class);
    private static final long idBase = System.currentTimeMillis();
    private static final AtomicInteger nextId = new AtomicInteger(1);
    static final int ENTRY_OVERHEAD_SIZE = 20;
    private final Map<UUID, Integer> cfLastWrite = new HashMap<UUID, Integer>();
    public final long id = CommitLogSegment.getNextId();
    private final File logFile;
    private final RandomAccessFile logFileAccessor;
    private boolean needsSync = false;
    private final MappedByteBuffer buffer;
    private boolean closed;
    public final CommitLogDescriptor descriptor = new CommitLogDescriptor(this.id);

    public static CommitLogSegment freshSegment() {
        return new CommitLogSegment(null);
    }

    public static long getNextId() {
        return idBase + (long)nextId.getAndIncrement();
    }

    CommitLogSegment(String filePath) {
        this.logFile = new File(DatabaseDescriptor.getCommitLogLocation(), this.descriptor.fileName());
        boolean isCreating = true;
        try {
            File oldFile;
            if (filePath != null && (oldFile = new File(filePath)).exists()) {
                logger.debug("Re-using discarded CommitLog segment for {} from {}", (Object)this.id, (Object)filePath);
                if (!oldFile.renameTo(this.logFile)) {
                    throw new IOException("Rename from " + filePath + " to " + this.id + " failed");
                }
                isCreating = false;
            }
            this.logFileAccessor = new RandomAccessFile(this.logFile, "rw");
            if (isCreating) {
                logger.debug("Creating new commit log segment {}", (Object)this.logFile.getPath());
            }
            this.logFileAccessor.setLength(DatabaseDescriptor.getCommitLogSegmentSize());
            this.buffer = this.logFileAccessor.getChannel().map(FileChannel.MapMode.READ_WRITE, 0L, DatabaseDescriptor.getCommitLogSegmentSize());
            this.buffer.putInt(0);
            this.buffer.position(0);
            this.needsSync = true;
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, this.logFile);
        }
    }

    public void discard(boolean deleteFile) {
        this.close();
        if (deleteFile) {
            FileUtils.deleteWithConfirm(this.logFile);
        }
    }

    public CommitLogSegment recycle() {
        this.buffer.position(0);
        this.buffer.putInt(0);
        this.buffer.position(0);
        try {
            this.sync();
        }
        catch (FSWriteError e) {
            logger.error("I/O error flushing " + this + " " + e);
            throw e;
        }
        this.close();
        return new CommitLogSegment(this.getPath());
    }

    public boolean hasCapacityFor(long size) {
        return size <= (long)this.buffer.remaining();
    }

    private void markDirty(RowMutation rowMutation, ReplayPosition repPos) {
        for (ColumnFamily columnFamily : rowMutation.getColumnFamilies()) {
            CFMetaData cfm = Schema.instance.getCFMetaData(columnFamily.id());
            if (cfm == null) {
                logger.error("Attempted to write commit log entry for unrecognized column family: " + columnFamily.id());
                continue;
            }
            this.markCFDirty(cfm.cfId, repPos.position);
        }
    }

    public ReplayPosition write(RowMutation rowMutation) throws IOException {
        assert (!this.closed);
        ReplayPosition repPos = this.getContext();
        this.markDirty(rowMutation, repPos);
        PureJavaCrc32 checksum = new PureJavaCrc32();
        byte[] serializedRow = FBUtilities.serialize(rowMutation, RowMutation.serializer, 6);
        checksum.update(serializedRow.length);
        this.buffer.putInt(serializedRow.length);
        this.buffer.putLong(checksum.getValue());
        this.buffer.put(serializedRow);
        checksum.update(serializedRow, 0, serializedRow.length);
        this.buffer.putLong(checksum.getValue());
        if (this.buffer.remaining() >= 4) {
            this.buffer.putInt(0);
            this.buffer.position(this.buffer.position() - 4);
        }
        this.needsSync = true;
        return repPos;
    }

    public void sync() {
        if (this.needsSync) {
            try {
                this.buffer.force();
            }
            catch (Exception e) {
                throw new FSWriteError((Throwable)e, this.getPath());
            }
            this.needsSync = false;
        }
    }

    public ReplayPosition getContext() {
        return new ReplayPosition(this.id, this.buffer.position());
    }

    public String getPath() {
        return this.logFile.getPath();
    }

    public String getName() {
        return this.logFile.getName();
    }

    public void close() {
        if (this.closed) {
            return;
        }
        try {
            FileUtils.clean(this.buffer);
            this.logFileAccessor.close();
            this.closed = true;
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, this.getPath());
        }
    }

    private void markCFDirty(UUID cfId, Integer position) {
        this.cfLastWrite.put(cfId, position);
    }

    public synchronized void markClean(UUID cfId, ReplayPosition context) {
        Integer lastWritten = this.cfLastWrite.get(cfId);
        if (!(lastWritten == null || this.contains(context) && lastWritten >= context.position)) {
            this.cfLastWrite.remove(cfId);
        }
    }

    public synchronized Collection<UUID> getDirtyCFIDs() {
        return new ArrayList<UUID>(this.cfLastWrite.keySet());
    }

    public synchronized boolean isUnused() {
        return this.cfLastWrite.isEmpty();
    }

    public boolean contains(ReplayPosition context) {
        return context.segment == this.id;
    }

    public String dirtyString() {
        StringBuilder sb = new StringBuilder();
        for (UUID cfId : this.getDirtyCFIDs()) {
            CFMetaData m = Schema.instance.getCFMetaData(cfId);
            sb.append(m == null ? "<deleted>" : m.cfName).append(" (").append(cfId).append("), ");
        }
        return sb.toString();
    }

    public String toString() {
        return "CommitLogSegment(" + this.getPath() + ')';
    }

    public int position() {
        return this.buffer.position();
    }

    public static class CommitLogSegmentFileComparator
    implements Comparator<File> {
        @Override
        public int compare(File f, File f2) {
            CommitLogDescriptor desc = CommitLogDescriptor.fromFileName(f.getName());
            CommitLogDescriptor desc2 = CommitLogDescriptor.fromFileName(f2.getName());
            return (int)(desc.id - desc2.id);
        }
    }
}

