/*
 * Copyright 2007 Werner Guttmann
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.castor.spring.xml;

import java.net.URL;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.exolab.castor.mapping.Mapping;
import org.exolab.castor.mapping.MappingException;
import org.exolab.castor.xml.XMLContext;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;
import org.xml.sax.InputSource;

/**
 * Factory bean to instantiate an {@link XMLContext} and pre-configure it with
 * the mappings found in the mapping files specified in the bean definition.
 * 
 * @author <a href="mailto:werner DOT guttmann AT gmx DOT net">Werner Guttmann</a>
 *
 * @since 1.2.1
 */
public class XMLContextFactoryBean implements FactoryBean, InitializingBean {

    /**
     * Log instance
     */
    private static final Log LOG = LogFactory
            .getLog(XMLContextFactoryBean.class);

    /**
     * Spring resource defining Castor properties
     */
    private Properties castorProperties = new Properties();

    /**
     * Spring resource defining mapping file locations
     */
    private List mappingLocations = new LinkedList();

    /**
     * {@link XMLContext} used to pre-configure Castor for
     * XML data binding.
     */
    private XMLContext xmlContext;
    
    /**
     * A list of qualified class names for which class descriptors will be loaded
     * from the class path. 
     * 
     * The use of this property is useful when no mapping is used, as happens when 
     * the domain classes have been generated using the XML code generator (in which 
     * case instead of a mapping file class descriptor files will be generated).
     */
    private List classes = new LinkedList();
    
    /**
     * A list of package names; for each package defined, all available class descriptors 
     * will be loaded from the class path.  
     * <p>
     * The use of this method is useful when no mapping is used, as happens when the 
     * domain classes have been generated using the XML code generator (in which case 
     * instead of a mapping file class descriptor files will be generated).
     * <p>
     * Please note that this will work <b>only</b> if you provide the <tt>.castor.cdr</tt>
     * file with your generated classes (as generated by the XML code generator).
     */
    private List packages = new LinkedList();

    /**
     * {@inheritDoc}
     * 
     * @see org.springframework.beans.factory.InitializingBean#afterPropertiesSet()
     */
    public void afterPropertiesSet() throws Exception {
        this.xmlContext = new XMLContext();
        
        // TODO[WG]: add code to load qualified class file descriptors
        if (!classes.isEmpty()) {
            // xmlContext.addClass
        }
        
        if (!packages.isEmpty()) {
            xmlContext.addPackages((String[]) packages.toArray());
        }
        
        if (!mappingLocations.isEmpty()) {
            String mappingLocation = null;
            Iterator iter = mappingLocations.iterator();
            try {
                Mapping mapping = new Mapping();
                while (iter.hasNext()) {
                    mappingLocation = (String) iter.next();
                    URL mappingResource = getClass().getClassLoader()
                            .getResource(mappingLocation);
                    mapping.loadMapping(new InputSource(mappingResource
                            .openStream()));
                }

                xmlContext.addMapping(mapping);
            } catch (MappingException e) {
                LOG.error(
                        "Problem locating/loading Castor mapping file from location "
                                + mappingLocation, e);
                throw e;
            }
        }
        
        // deal with properties as defined in the spring application context
        if (!castorProperties.isEmpty()) {
            Set entries = castorProperties.entrySet();
            Iterator iterator = entries.iterator();
            while (iterator.hasNext()) {
                Map.Entry entry = (Entry) iterator.next();
                xmlContext.setProperty((String) entry.getKey(), entry.getValue());
            }
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @see org.springframework.beans.factory.FactoryBean#getObject()
     */
    public Object getObject() throws Exception {
        return this.xmlContext;
    }

    /**
     * {@inheritDoc}
     * 
     * @see org.springframework.beans.factory.FactoryBean#getObjectType()
     */
    public Class getObjectType() {
        if (this.xmlContext == null) {
            return XMLContext.class;
        }

        return this.xmlContext.getClass();
    }

    /**
     * {@inheritDoc}
     * 
     * @see org.springframework.beans.factory.FactoryBean#isSingleton()
     */
    public boolean isSingleton() {
        return true;
    }

    public void setCastorProperties(Properties castorProperties) {
        this.castorProperties.putAll(castorProperties);
    }

    /**
     * Sets a collection of mapping (file) locations.
     * 
     * @param mappingLocations
     *            A collection of mapping (file) locations.
     */
    public void setMappingLocations(final List mappingLocations) {
        this.mappingLocations.addAll(mappingLocations);
    }
    
    /**
     * Sets a list of qualified class names for which class descriptors will be loaded
     * from the class path. 
     * 
     * The use of this property is useful when no mapping is used, as happens when 
     * the domain classes have been generated using the XML code generator (in which 
     * case instead of a mapping file class descriptor files will be generated).
     * 
     * @param classes A list of qualified class names for which class descriptors will be loaded
     * from the class path.
     */
    public void setClasses(final List classes) {
        this.classes.addAll(classes);
    }

    /**
     * A list of package names; for each package defined, all available class descriptors 
     * will be loaded from the class path.  
     * <p>
     * The use of this method is useful when no mapping is used, as happens when the 
     * domain classes have been generated using the XML code generator (in which case 
     * instead of a mapping file class descriptor files will be generated).
     * <p>
     * Please note that this will work <b>only</b> if you provide the <tt>.castor.cdr</tt>
     * file with your generated classes (as generated by the XML code generator).
     * 
     * @param packages A list of package names
     */
    public void setPackages(final List packages) {
        this.packages.addAll(packages);
    }


}