/*
 * Bibliothek - DockingFrames
 * Library built on Java/Swing, allows the user to "drag and drop"
 * panels containing any Swing-Component the developer likes to add.
 * 
 * Copyright (C) 2009 Benjamin Sigg
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * Benjamin Sigg
 * benjamin_sigg@gmx.ch
 * CH - Switzerland
 */
package bibliothek.gui.dock.facile.mode;

import java.util.HashMap;
import java.util.Map;

import bibliothek.gui.Dockable;
import bibliothek.gui.dock.action.DockActionSource;
import bibliothek.gui.dock.support.mode.Mode;

/**
 * This {@link LocationModeActionProvider} mimics a map with keys managed by 
 * a subclass. In its {@link #getActions(Dockable, Mode, DockActionSource)}
 * method it selects the one provider whose key matches the current mode to provide the actions. 
 * @author Benjamin Sigg
 */
public abstract class MappingLocationModeActionProvider implements LocationModeActionProvider {
	/** tells for each dockable which provider was used last */
	private Map<Dockable, LocationModeActionProvider> lastUsedProviders = new HashMap<Dockable, LocationModeActionProvider>();
	
	public void destroy( Dockable dockable, DockActionSource source ){
		LocationModeActionProvider provider = lastUsedProviders.remove( dockable );
		if( provider != null ){
			provider.destroy( dockable, source );
		}
	}
	
	public DockActionSource getActions( Dockable dockable, Mode<Location> currentMode, DockActionSource currentSource ){
		LocationModeActionProvider provider = getProvider( dockable, currentMode, currentSource );
		if( provider == null ){
			lastUsedProviders.remove( dockable );
			return null;
		}
		else{
			lastUsedProviders.put( dockable, provider );
			return provider.getActions( dockable, currentMode, currentSource );
		}
	}
	
	/**
	 * Gets the {@link LocationModeActionProvider} which fits the key generated by the arguments
	 * of this methode.
	 * @param dockable the element for which the actions will be used
	 * @param currentMode the current mode of <code>dockable</code>
	 * @param currentSource the current actions of <code>dockable</code>
	 * @return the provider or <code>null</code>
	 */
	protected abstract LocationModeActionProvider getProvider( Dockable dockable, Mode<Location> currentMode, DockActionSource currentSource );
}
