/// <reference path="../libs/hawtio-utilities/defs.d.ts"/>

/// <reference path="../../includes.ts"/>
var HawtioKeycloak;
(function (HawtioKeycloak) {
    HawtioKeycloak.pluginName = 'hawtio-keycloak';
    HawtioKeycloak.log = Logger.get(HawtioKeycloak.pluginName);
    HawtioKeycloak.keycloak = undefined;
})(HawtioKeycloak || (HawtioKeycloak = {}));

/// <reference path="keycloakGlobals.ts"/>

/// <reference path="keycloakGlobals.ts"/>
/// <reference path="keycloakHelpers.ts"/>
var HawtioKeycloak;
(function (HawtioKeycloak) {
    HawtioKeycloak._module = angular.module(HawtioKeycloak.pluginName, []);
    var userProfile = undefined;
    hawtioPluginLoader.addModule(HawtioKeycloak.pluginName);
    HawtioKeycloak._module.config(['$provide', '$httpProvider', function ($provide, $httpProvider) {
        $provide.decorator('userDetails', ['$delegate', function ($delegate) {
            if (userProfile) {
                return _.merge($delegate, userProfile, {
                    logout: function () {
                        if (userProfile && HawtioKeycloak.keycloak) {
                            HawtioKeycloak.keycloak.logout();
                        }
                    }
                });
            }
            else {
                return $delegate;
            }
        }]);
        // only add the itnerceptor if we have keycloak otherwise
        // we'll get an undefined exception in the interceptor
        if (HawtioKeycloak.keycloak) {
            $httpProvider.interceptors.push(AuthInterceptorService.Factory);
        }
    }]);
    HawtioKeycloak._module.run(['userDetails', 'Idle', '$rootScope', function (userDetails, Idle, $rootScope) {
        // log.debug("loaded, userDetails: ", userDetails);
        Idle.watch();
        $rootScope.$on('IdleTimeout', function () {
            // let the end application handle this event
            // userDetails.logout();
        });
        $rootScope.$on('Keepalive', function () {
            var keycloak = HawtioKeycloak.keycloak;
            if (keycloak) {
                keycloak.updateToken(30);
            }
        });
    }]);
    hawtioPluginLoader.registerPreBootstrapTask(function (next) {
        if (!window['KeycloakConfig']) {
            HawtioKeycloak.log.debug("Keycloak disabled");
            next();
            return;
        }
        var keycloak = HawtioKeycloak.keycloak = Keycloak(KeycloakConfig);
        keycloak.init().success(function (authenticated) {
            HawtioKeycloak.log.debug("Authenticated: ", authenticated);
            if (!authenticated) {
                keycloak.login({
                    redirectUri: window.location.href,
                });
            }
            else {
                keycloak.loadUserProfile().success(function (profile) {
                    userProfile = profile;
                    userProfile.token = keycloak.token;
                    next();
                }).error(function () {
                    HawtioKeycloak.log.debug("Failed to load user profile");
                    next();
                });
            }
        }).error(function () {
            HawtioKeycloak.log.debug("Failed to initialize Keycloak, token unavailable");
            next();
        });
    });
    var AuthInterceptorService = (function () {
        function AuthInterceptorService($q) {
            var _this = this;
            this.$q = $q;
            this.request = function (request) {
                var addBearer, deferred;
                addBearer = function () {
                    var keycloak = HawtioKeycloak.keycloak;
                    return keycloak.updateToken(5).success(function () {
                        var token = HawtioKeycloak.keycloak.token;
                        request.headers.Authorization = 'Bearer ' + token;
                        deferred.notify();
                        return deferred.resolve(request);
                    }).error(function () {
                        console.log("Couldn't update token");
                    });
                };
                deferred = _this.$q.defer();
                addBearer();
                return _this.$q.when(deferred.promise);
            };
            this.responseError = function (rejection) {
                if (rejection.status === 401) {
                    HawtioKeycloak.keycloak.logout();
                }
                return _this.$q.reject(rejection);
            };
        }
        AuthInterceptorService.Factory = function ($q) {
            return new AuthInterceptorService($q);
        };
        AuthInterceptorService.$inject = ['$q'];
        return AuthInterceptorService;
    })();
    HawtioKeycloak._module.requires.push("ngIdle");
})(HawtioKeycloak || (HawtioKeycloak = {}));

/// <reference path="../../includes.ts"/>
var OSOAuth;
(function (OSOAuth) {
    OSOAuth.pluginName = 'hawtio-os-oauth';
    OSOAuth.log = Logger.get(OSOAuth.pluginName);
})(OSOAuth || (OSOAuth = {}));

/// <reference path="osOAuthGlobals.ts"/>
var OSOAuth;
(function (OSOAuth) {
    var OS_TOKEN_STORAGE_KEY = 'osAuthCreds';
    function authenticatedHttpRequest(options, userDetails) {
        return $.ajax(_.extend(options, {
            beforeSend: function (request) {
                if (userDetails.token) {
                    request.setRequestHeader('Authorization', 'Bearer ' + userDetails.token);
                }
            }
        }));
    }
    OSOAuth.authenticatedHttpRequest = authenticatedHttpRequest;
    function doLogout(config, userDetails) {
        var currentURI = new URI(window.location.href);
        var uri = new URI(config.oauth_authorize_uri);
        uri.path('/osapi/v1beta1/oAuthAccessTokens' + userDetails.token);
        authenticatedHttpRequest({
            type: 'DELETE',
            url: uri.toString()
        }, userDetails).always(function () {
            clearTokenStorage();
            doLogin(OSOAuthConfig, {
                uri: currentURI.toString()
            });
        });
    }
    OSOAuth.doLogout = doLogout;
    function doLogin(config, options) {
        var clientId = config.oauth_client_id;
        var targetURI = config.oauth_authorize_uri;
        var uri = new URI(targetURI);
        uri.query({
            client_id: clientId,
            response_type: 'token',
            state: options.uri,
            redirect_uri: options.uri
        });
        var target = uri.toString();
        OSOAuth.log.debug("Redirecting to URI: ", target);
        window.location.href = target;
    }
    OSOAuth.doLogin = doLogin;
    function extractToken(uri) {
        var query = uri.query(true);
        OSOAuth.log.debug("Query: ", query);
        var fragmentParams = new URI("?" + uri.fragment()).query(true);
        OSOAuth.log.debug("FragmentParams: ", fragmentParams);
        if (fragmentParams.access_token && fragmentParams.token_type === "bearer") {
            OSOAuth.log.debug("Got token");
            var localStorage = Core.getLocalStorage();
            var creds = {
                token_type: fragmentParams.token_type,
                access_token: fragmentParams.access_token,
                expires_in: fragmentParams.expires_in
            };
            localStorage['osAuthCreds'] = angular.toJson(creds);
            delete fragmentParams.token_type;
            delete fragmentParams.access_token;
            delete fragmentParams.expires_in;
            uri.fragment("").query(fragmentParams);
            var target = uri.toString();
            OSOAuth.log.debug("redirecting to: ", target);
            window.location.href = target;
            return creds;
        }
        else {
            OSOAuth.log.debug("No token in URI");
            return undefined;
        }
    }
    OSOAuth.extractToken = extractToken;
    function clearTokenStorage() {
        var localStorage = Core.getLocalStorage();
        delete localStorage[OS_TOKEN_STORAGE_KEY];
    }
    OSOAuth.clearTokenStorage = clearTokenStorage;
    function checkToken(uri) {
        var localStorage = Core.getLocalStorage();
        var answer = undefined;
        if (OS_TOKEN_STORAGE_KEY in localStorage) {
            try {
                answer = angular.fromJson(localStorage[OS_TOKEN_STORAGE_KEY]);
            }
            catch (e) {
                clearTokenStorage();
                // must be broken...
                OSOAuth.log.debug("Error extracting osAuthCreds value: ", e);
            }
        }
        if (!answer) {
            answer = extractToken(uri);
        }
        OSOAuth.log.debug("Using creds: ", answer);
        return answer;
    }
    OSOAuth.checkToken = checkToken;
})(OSOAuth || (OSOAuth = {}));

/// <reference path="osOAuthHelpers.ts"/>
var OSOAuth;
(function (OSOAuth) {
    OSOAuth._module = angular.module(OSOAuth.pluginName, []);
    var userProfile = undefined;
    hawtioPluginLoader.addModule(OSOAuth.pluginName);
    OSOAuth._module.config(['$provide', function ($provide) {
        $provide.decorator('userDetails', ['$delegate', function ($delegate) {
            if (userProfile) {
                return _.merge($delegate, userProfile, {
                    username: userProfile.fullName,
                    logout: function () {
                        OSOAuth.doLogout(OSOAuthConfig, userProfile);
                    }
                });
            }
            else {
                return $delegate;
            }
        }]);
    }]);
    OSOAuth._module.config(['$httpProvider', function ($httpProvider) {
        if (userProfile && userProfile.token) {
            $httpProvider.defaults.headers.common = {
                'Authorization': 'Bearer ' + userProfile.token
            };
        }
    }]);
    OSOAuth._module.run(['userDetails', function (userDetails) {
        // log.debug("loaded, userDetails: ", userDetails);
    }]);
    hawtioPluginLoader.registerPreBootstrapTask(function (next) {
        if (!window['OSOAuthConfig']) {
            OSOAuth.log.debug("oauth disabled");
            next();
            return;
        }
        if (!OSOAuthConfig.oauth_client_id || !OSOAuthConfig.oauth_authorize_uri) {
            OSOAuth.log.debug("Invalid oauth config, disabled oauth");
            next();
            return;
        }
        OSOAuth.log.debug("config: ", OSOAuthConfig);
        var currentURI = new URI(window.location.href);
        var fragmentParams = OSOAuth.checkToken(currentURI);
        if (fragmentParams) {
            var tmp = {
                token: fragmentParams.access_token,
                expiry: fragmentParams.expires_in,
                type: fragmentParams.token_type
            };
            var uri = new URI(OSOAuthConfig.oauth_authorize_uri);
            uri.path('/osapi/v1beta1/users/~');
            OSOAuth.authenticatedHttpRequest({
                type: 'GET',
                url: uri.toString(),
            }, tmp).done(function (response) {
                userProfile = {};
                _.extend(userProfile, tmp, response);
            }).fail(function () {
                OSOAuth.clearTokenStorage();
                OSOAuth.doLogin(OSOAuthConfig, {
                    uri: currentURI.toString()
                });
            }).always(function () {
                next();
            });
        }
        else {
            OSOAuth.clearTokenStorage();
            OSOAuth.doLogin(OSOAuthConfig, {
                uri: currentURI.toString()
            });
        }
    });
})(OSOAuth || (OSOAuth = {}));
