/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.lifecycle;

import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.cassandra.db.lifecycle.LogRecord;
import org.apache.cassandra.db.lifecycle.LogReplica;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.utils.Throwables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LogReplicaSet {
    private static final Logger logger = LoggerFactory.getLogger(LogReplicaSet.class);
    private final Map<File, LogReplica> replicasByFile = new LinkedHashMap<File, LogReplica>();

    private Collection<LogReplica> replicas() {
        return this.replicasByFile.values();
    }

    void addReplicas(List<File> replicas) {
        replicas.forEach(this::addReplica);
    }

    void addReplica(File file) {
        File folder = file.getParentFile();
        assert (!this.replicasByFile.containsKey(folder));
        this.replicasByFile.put(folder, LogReplica.open(file));
        if (logger.isTraceEnabled()) {
            logger.trace("Added log file replica {} ", (Object)file);
        }
    }

    void maybeCreateReplica(File folder, String fileName, Set<LogRecord> records) {
        if (this.replicasByFile.containsKey(folder)) {
            return;
        }
        LogReplica replica = LogReplica.create(folder, fileName);
        records.forEach(replica::append);
        this.replicasByFile.put(folder, replica);
        if (logger.isTraceEnabled()) {
            logger.trace("Created new file replica {}", (Object)replica);
        }
    }

    Throwable syncFolder(Throwable accumulate) {
        return Throwables.perform(accumulate, this.replicas().stream().map(s -> s::syncFolder));
    }

    Throwable delete(Throwable accumulate) {
        return Throwables.perform(accumulate, this.replicas().stream().map(s -> s::delete));
    }

    private static boolean isPrefixMatch(String first, String second) {
        return first.length() >= second.length() ? first.startsWith(second) : second.startsWith(first);
    }

    boolean readRecords(Set<LogRecord> records) {
        Map linesByReplica = this.replicas().stream().map(LogReplica::file).collect(Collectors.toMap(Function.identity(), FileUtils::readLines));
        int maxNumLines = linesByReplica.values().stream().map(List::size).reduce(0, Integer::max);
        for (int i = 0; i < maxNumLines; ++i) {
            String firstLine = null;
            boolean partial = false;
            for (Map.Entry entry : linesByReplica.entrySet()) {
                List currentLines = entry.getValue();
                if (i >= currentLines.size()) continue;
                String currentLine = (String)currentLines.get(i);
                if (firstLine == null) {
                    firstLine = currentLine;
                    continue;
                }
                if (!LogReplicaSet.isPrefixMatch(firstLine, currentLine)) {
                    logger.error("Mismatched line in file {}: got '{}' expected '{}', giving up", ((File)entry.getKey()).getName(), currentLine, firstLine);
                    return false;
                }
                if (firstLine.equals(currentLine)) continue;
                if (i == currentLines.size() - 1) {
                    logger.warn("Mismatched last line in file {}: '{}' not the same as '{}'", ((File)entry.getKey()).getName(), currentLine, firstLine);
                    if (currentLine.length() > firstLine.length()) {
                        firstLine = currentLine;
                    }
                    partial = true;
                    continue;
                }
                logger.error("Mismatched line in file {}: got '{}' expected '{}', giving up", ((File)entry.getKey()).getName(), currentLine, firstLine);
                return false;
            }
            LogRecord record = LogRecord.make(firstLine);
            if (records.contains(record)) {
                logger.error("Found duplicate record {} for {}, giving up", (Object)record, (Object)record.fileName());
                return false;
            }
            if (partial) {
                record.setPartial();
            }
            records.add(record);
            if (!record.isFinal() || i == maxNumLines - 1) continue;
            logger.error("Found too many lines for {}, giving up", (Object)record.fileName());
            return false;
        }
        return true;
    }

    void append(LogRecord record) {
        Throwable err = Throwables.perform(null, this.replicas().stream().map(r -> () -> r.append(record)));
        if (err != null) {
            if (!record.isFinal() || err.getSuppressed().length == this.replicas().size() - 1) {
                Throwables.maybeFail(err);
            }
            logger.error("Failed to add record '{}' to some replicas '{}'", (Object)record, (Object)this);
        }
    }

    boolean exists() {
        Optional<Boolean> ret = this.replicas().stream().map(LogReplica::exists).reduce(Boolean::logicalAnd);
        return ret.isPresent() ? ret.get() : false;
    }

    void close() {
        Throwables.maybeFail(Throwables.perform(null, this.replicas().stream().map(r -> r::close)));
    }

    public String toString() {
        Optional<String> ret = this.replicas().stream().map(LogReplica::toString).reduce(String::concat);
        return ret.isPresent() ? ret.get() : "[-]";
    }

    @VisibleForTesting
    List<File> getFiles() {
        return this.replicas().stream().map(LogReplica::file).collect(Collectors.toList());
    }

    @VisibleForTesting
    List<String> getFilePaths() {
        return this.replicas().stream().map(LogReplica::file).map(File::getPath).collect(Collectors.toList());
    }
}

