/*
 * Decompiled with CFR 0.152.
 */
package org.hawkular.metrics.schema;

import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.Host;
import com.datastax.driver.core.JdkSSLOptions;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.exceptions.NoHostAvailableException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import javax.net.ssl.SSLContext;
import org.hawkular.metrics.scheduler.api.JobsManager;
import org.hawkular.metrics.schema.SchemaService;
import org.hawkular.metrics.schema.VersionUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Installer {
    private static final Logger logger = LoggerFactory.getLogger(Installer.class);
    private List<String> cassandraNodes;
    private int cassandraConnectionMaxRetries;
    private long cassandraConnectionMaxDelay;
    private boolean useSSL;
    private int cqlPort = Integer.getInteger("hawkular.metrics.cassandra.cql-port", 9042);
    private String keyspace;
    private boolean resetdb;
    private int replicationFactor;
    private long versionUpdateDelay;
    private int versionUpdateMaxRetries;

    public Installer() {
        this.useSSL = Boolean.getBoolean("hawkular.metrics.cassandra.use-ssl");
        String nodes = System.getProperty("hawkular.metrics.cassandra.nodes", "127.0.0.1");
        this.cassandraNodes = new ArrayList<String>();
        Arrays.stream(nodes.split(",")).forEach(this.cassandraNodes::add);
        this.cassandraConnectionMaxDelay = Long.getLong("hawkular.metrics.cassandra.connection.max-delay", 30L) * 1000L;
        this.cassandraConnectionMaxRetries = Integer.getInteger("hawkular.metrics.cassandra.connection.max-retries", 5);
        this.keyspace = System.getProperty("hawkular.metrics.cassandra.keyspace", "hawkular_metrics");
        this.resetdb = Boolean.getBoolean("hawkular.metrics.cassandra.resetdb");
        this.replicationFactor = Integer.getInteger("hawkular.metrics.cassandra.replication-factor", 1);
        this.versionUpdateDelay = Long.getLong("hawkular.metrics.version-update.delay", 5L) * 1000L;
        this.versionUpdateMaxRetries = Integer.getInteger("hawkular.metrics.version-update.max-retries", 10);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void run() {
        this.logVersion();
        this.logInstallerProperties();
        try (Session session = this.initSession();){
            this.waitForAllNodesToBeUp(session);
            SchemaService schemaService = new SchemaService();
            schemaService.run(session, this.keyspace, this.resetdb, this.replicationFactor, false);
            JobsManager jobsManager = new JobsManager(session);
            jobsManager.installJobs();
            schemaService.updateVersion(session, this.keyspace, this.versionUpdateDelay, this.versionUpdateMaxRetries);
            logger.info("Finished installation");
        }
        catch (InterruptedException e) {
            logger.warn("Aborting installation");
            System.exit(1);
        }
        catch (Exception e) {
            logger.warn("Installation failed", e);
            System.exit(1);
        }
        finally {
            System.exit(0);
        }
    }

    private void logVersion() {
        logger.info("Hawkular Metrics Schema Installer v{}", (Object)VersionUtil.getVersion());
    }

    private void logInstallerProperties() {
        logger.info("Configured installer properties:\n\tcqlPort = " + this.cqlPort + "\n\tuseSSL = " + this.useSSL + "\n\tcassandraNodes = " + this.cassandraNodes + "\n\tcassandraConnectionMaxDelay = " + this.cassandraConnectionMaxDelay + "\n\tcassandraConnectionMaxRetries = " + this.cassandraConnectionMaxRetries + "\n\tkeyspace = " + this.keyspace + "\n\tresetdb = " + this.resetdb + "\n\treplicationFactor = " + this.replicationFactor + "\n\tversionUpdateDelay = " + this.versionUpdateDelay + "\n\tversionUpdateMaxRetries = " + this.versionUpdateMaxRetries);
    }

    private Session initSession() throws InterruptedException {
        long retry = 5000L;
        while (true) {
            try {
                return this.createSession();
            }
            catch (NoHostAvailableException e) {
                logger.info("Cassandra may not be up yet. Retrying in {} ms", (Object)retry);
                Thread.sleep(retry);
                continue;
            }
            break;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Session createSession() {
        Cluster.Builder clusterBuilder = new Cluster.Builder();
        clusterBuilder.addContactPoints(this.cassandraNodes.toArray(new String[0]));
        if (this.useSSL) {
            JdkSSLOptions sslOptions = null;
            try {
                String[] defaultCipherSuites = new String[]{"TLS_RSA_WITH_AES_128_CBC_SHA", "TLS_RSA_WITH_AES_256_CBC_SHA"};
                sslOptions = JdkSSLOptions.builder().withSSLContext(SSLContext.getDefault()).withCipherSuites(defaultCipherSuites).build();
                clusterBuilder.withSSL(sslOptions);
            }
            catch (NoSuchAlgorithmException e) {
                throw new RuntimeException("SSL support is required but is not available in the JVM.", e);
            }
        }
        clusterBuilder.withoutJMXReporting();
        Cluster cluster = clusterBuilder.build();
        cluster.init();
        Session createdSession = null;
        try {
            Session session = createdSession = cluster.connect("system");
            return session;
        }
        finally {
            if (createdSession == null) {
                cluster.close();
            }
        }
    }

    private void waitForAllNodesToBeUp(Session session) {
        boolean isReady = false;
        int attempts = this.cassandraConnectionMaxRetries;
        long delay = 2000L;
        block2: while (!isReady && !Thread.currentThread().isInterrupted() && attempts-- >= 0) {
            isReady = true;
            for (Host host : session.getCluster().getMetadata().getAllHosts()) {
                if (host.isUp()) continue;
                isReady = false;
                logger.warn("Cassandra node {} may not be up yet. Waiting {} ms for node to come up", (Object)host, (Object)delay);
                try {
                    Thread.sleep(delay);
                    delay = Math.min(delay * 2L, this.cassandraConnectionMaxDelay);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
                continue block2;
            }
        }
        if (!isReady) {
            throw new RuntimeException("It appears that not all nodes in the Cassandra cluster are up after " + attempts + " checks. Schema updates cannot proceed without all nodes being up.");
        }
    }

    public static void main(String[] args) {
        new Installer().run();
    }
}

