/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.core.dns.impl.fix;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufHolder;
import io.netty.channel.AddressedEnvelope;
import io.netty.channel.socket.InternetProtocolFamily;
import io.netty.handler.codec.CorruptedFrameException;
import io.netty.handler.codec.dns.DefaultDnsQuestion;
import io.netty.handler.codec.dns.DnsQuestion;
import io.netty.handler.codec.dns.DnsRawRecord;
import io.netty.handler.codec.dns.DnsRecordType;
import io.netty.handler.codec.dns.DnsResponse;
import io.netty.handler.codec.dns.DnsResponseCode;
import io.netty.handler.codec.dns.DnsSection;
import io.netty.resolver.dns.DnsCache;
import io.netty.resolver.dns.DnsCacheEntry;
import io.netty.resolver.dns.DnsServerAddressStream;
import io.netty.util.CharsetUtil;
import io.netty.util.ReferenceCountUtil;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.FutureListener;
import io.netty.util.concurrent.GenericFutureListener;
import io.netty.util.concurrent.Promise;
import io.netty.util.internal.StringUtil;
import io.vertx.core.dns.impl.fix.DnsNameResolver;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

abstract class DnsNameResolverContext<T> {
    private static final int INADDRSZ4 = 4;
    private static final int INADDRSZ6 = 16;
    private static final FutureListener<AddressedEnvelope<DnsResponse, InetSocketAddress>> RELEASE_RESPONSE = new FutureListener<AddressedEnvelope<DnsResponse, InetSocketAddress>>(){

        @Override
        public void operationComplete(Future<AddressedEnvelope<DnsResponse, InetSocketAddress>> future) {
            if (future.isSuccess()) {
                future.getNow().release();
            }
        }
    };
    private final DnsNameResolver parent;
    private final DnsServerAddressStream nameServerAddrs;
    private final Promise<T> promise;
    private final String hostname;
    private final DnsCache resolveCache;
    private final boolean traceEnabled;
    private final int maxAllowedQueries;
    private final InternetProtocolFamily[] resolveAddressTypes;
    private final Set<Future<AddressedEnvelope<DnsResponse, InetSocketAddress>>> queriesInProgress = Collections.newSetFromMap(new IdentityHashMap());
    private List<DnsCacheEntry> resolvedEntries;
    private StringBuilder trace;
    private int allowedQueries;
    private boolean triedCNAME;

    protected DnsNameResolverContext(DnsNameResolver parent, String hostname, Promise<T> promise, DnsCache resolveCache) {
        this.parent = parent;
        this.promise = promise;
        this.hostname = hostname;
        this.resolveCache = resolveCache;
        this.nameServerAddrs = parent.nameServerAddresses.stream();
        this.maxAllowedQueries = parent.maxQueriesPerResolve();
        this.resolveAddressTypes = parent.resolveAddressTypesUnsafe();
        this.traceEnabled = parent.isTraceEnabled();
        this.allowedQueries = this.maxAllowedQueries;
    }

    protected Promise<T> promise() {
        return this.promise;
    }

    void resolve() {
        InetSocketAddress nameServerAddrToTry = this.nameServerAddrs.next();
        for (InternetProtocolFamily f : this.resolveAddressTypes) {
            DnsRecordType type;
            switch (f) {
                case IPv4: {
                    type = DnsRecordType.A;
                    break;
                }
                case IPv6: {
                    type = DnsRecordType.AAAA;
                    break;
                }
                default: {
                    throw new Error();
                }
            }
            this.query(nameServerAddrToTry, new DefaultDnsQuestion(this.hostname, type));
        }
    }

    private void query(InetSocketAddress nameServerAddr, final DnsQuestion question) {
        if (this.allowedQueries == 0 || this.promise.isCancelled()) {
            this.tryToFinishResolve();
            return;
        }
        --this.allowedQueries;
        Future<AddressedEnvelope<DnsResponse, InetSocketAddress>> f = this.parent.query(nameServerAddr, question);
        this.queriesInProgress.add(f);
        f.addListener((GenericFutureListener<Future<AddressedEnvelope<DnsResponse, InetSocketAddress>>>)new FutureListener<AddressedEnvelope<DnsResponse, InetSocketAddress>>(){

            @Override
            public void operationComplete(Future<AddressedEnvelope<DnsResponse, InetSocketAddress>> future) {
                DnsNameResolverContext.this.queriesInProgress.remove(future);
                if (DnsNameResolverContext.this.promise.isDone() || future.isCancelled()) {
                    return;
                }
                try {
                    if (future.isSuccess()) {
                        DnsNameResolverContext.this.onResponse(question, future.getNow());
                    } else {
                        if (DnsNameResolverContext.this.traceEnabled) {
                            DnsNameResolverContext.this.addTrace(future.cause());
                        }
                        DnsNameResolverContext.this.query(DnsNameResolverContext.this.nameServerAddrs.next(), question);
                    }
                }
                finally {
                    DnsNameResolverContext.this.tryToFinishResolve();
                }
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void onResponse(DnsQuestion question, AddressedEnvelope<DnsResponse, InetSocketAddress> envelope) {
        try {
            DnsResponse res = envelope.content();
            DnsResponseCode code = res.code();
            if (code == DnsResponseCode.NOERROR) {
                DnsRecordType type = question.type();
                if (type == DnsRecordType.A || type == DnsRecordType.AAAA) {
                    this.onResponseAorAAAA(type, question, envelope);
                } else if (type == DnsRecordType.CNAME) {
                    this.onResponseCNAME(question, envelope);
                }
                return;
            }
            if (this.traceEnabled) {
                this.addTrace(envelope.sender(), "response code: " + code + " with " + res.count(DnsSection.ANSWER) + " answer(s) and " + res.count(DnsSection.AUTHORITY) + " authority resource(s)");
            }
            if (code != DnsResponseCode.NXDOMAIN) {
                this.query(this.nameServerAddrs.next(), question);
            }
        }
        finally {
            ReferenceCountUtil.safeRelease(envelope);
        }
    }

    private void onResponseAorAAAA(DnsRecordType qType, DnsQuestion question, AddressedEnvelope<DnsResponse, InetSocketAddress> envelope) {
        DnsResponse response = envelope.content();
        Map<String, String> cnames = DnsNameResolverContext.buildAliasMap(response);
        int answerCount = response.count(DnsSection.ANSWER);
        boolean found = false;
        for (int i = 0; i < answerCount; ++i) {
            InetAddress resolved;
            ByteBuf content;
            int contentLen;
            Object r = response.recordAt(DnsSection.ANSWER, i);
            DnsRecordType type = r.type();
            if (type != DnsRecordType.A && type != DnsRecordType.AAAA) continue;
            String qName = question.name().toLowerCase(Locale.US);
            String rName = r.name().toLowerCase(Locale.US);
            if (!rName.equals(qName)) {
                String resolved2 = qName;
                while (!rName.equals(resolved2 = cnames.get(resolved2)) && resolved2 != null) {
                }
                if (resolved2 == null) continue;
            }
            if (!(r instanceof DnsRawRecord) || (contentLen = (content = ((ByteBufHolder)r).content()).readableBytes()) != 4 && contentLen != 16) continue;
            byte[] addrBytes = new byte[contentLen];
            content.getBytes(content.readerIndex(), addrBytes);
            try {
                resolved = InetAddress.getByAddress(this.hostname, addrBytes);
            }
            catch (UnknownHostException e) {
                throw new Error(e);
            }
            if (this.resolvedEntries == null) {
                this.resolvedEntries = new ArrayList<DnsCacheEntry>(8);
            }
            DnsCacheEntry e = new DnsCacheEntry(this.hostname, resolved);
            this.resolveCache.cache(this.hostname, resolved, r.timeToLive(), this.parent.ch.eventLoop());
            this.resolvedEntries.add(e);
            found = true;
        }
        if (found) {
            return;
        }
        if (this.traceEnabled) {
            this.addTrace(envelope.sender(), "no matching " + qType + " record found");
        }
        if (!cnames.isEmpty()) {
            this.onResponseCNAME(question, envelope, cnames, false);
        }
    }

    private void onResponseCNAME(DnsQuestion question, AddressedEnvelope<DnsResponse, InetSocketAddress> envelope) {
        this.onResponseCNAME(question, envelope, DnsNameResolverContext.buildAliasMap(envelope.content()), true);
    }

    private void onResponseCNAME(DnsQuestion question, AddressedEnvelope<DnsResponse, InetSocketAddress> response, Map<String, String> cnames, boolean trace) {
        String next;
        String name;
        String resolved = name = question.name().toLowerCase(Locale.US);
        boolean found = false;
        while (!cnames.isEmpty() && (next = cnames.remove(resolved)) != null) {
            found = true;
            resolved = next;
        }
        if (found) {
            this.followCname(response.sender(), name, resolved);
        } else if (trace && this.traceEnabled) {
            this.addTrace(response.sender(), "no matching CNAME record found");
        }
    }

    private static Map<String, String> buildAliasMap(DnsResponse response) {
        int answerCount = response.count(DnsSection.ANSWER);
        Map<String, String> cnames = null;
        for (int i = 0; i < answerCount; ++i) {
            ByteBuf recordContent;
            String domainName;
            Object r = response.recordAt(DnsSection.ANSWER, i);
            DnsRecordType type = r.type();
            if (type != DnsRecordType.CNAME || !(r instanceof DnsRawRecord) || (domainName = DnsNameResolverContext.decodeDomainName(recordContent = ((ByteBufHolder)r).content())) == null) continue;
            if (cnames == null) {
                cnames = new HashMap<String, String>();
            }
            cnames.put(r.name().toLowerCase(Locale.US), domainName.toLowerCase(Locale.US));
        }
        return cnames != null ? cnames : Collections.emptyMap();
    }

    void tryToFinishResolve() {
        if (!this.queriesInProgress.isEmpty()) {
            if (this.gotPreferredAddress()) {
                this.finishResolve();
            }
            return;
        }
        if (this.resolvedEntries == null && !this.triedCNAME) {
            this.triedCNAME = true;
            this.query(this.nameServerAddrs.next(), new DefaultDnsQuestion(this.hostname, DnsRecordType.CNAME));
            return;
        }
        this.finishResolve();
    }

    private boolean gotPreferredAddress() {
        if (this.resolvedEntries == null) {
            return false;
        }
        int size = this.resolvedEntries.size();
        switch (this.resolveAddressTypes[0]) {
            case IPv4: {
                for (int i = 0; i < size; ++i) {
                    if (!(this.resolvedEntries.get(i).address() instanceof Inet4Address)) continue;
                    return true;
                }
                break;
            }
            case IPv6: {
                for (int i = 0; i < size; ++i) {
                    if (!(this.resolvedEntries.get(i).address() instanceof Inet6Address)) continue;
                    return true;
                }
                break;
            }
        }
        return false;
    }

    private void finishResolve() {
        if (!this.queriesInProgress.isEmpty()) {
            InternetProtocolFamily[] i = this.queriesInProgress.iterator();
            while (i.hasNext()) {
                Future f = (Future)i.next();
                i.remove();
                if (f.cancel(false)) continue;
                f.addListener(RELEASE_RESPONSE);
            }
        }
        if (this.resolvedEntries != null) {
            for (InternetProtocolFamily f : this.resolveAddressTypes) {
                if (!this.finishResolve(f.addressType(), this.resolvedEntries)) continue;
                return;
            }
        }
        int tries = this.maxAllowedQueries - this.allowedQueries;
        StringBuilder buf = new StringBuilder(64);
        buf.append("failed to resolve '").append(this.hostname).append('\'');
        if (tries > 1) {
            if (tries < this.maxAllowedQueries) {
                buf.append(" after ").append(tries).append(" queries ");
            } else {
                buf.append(". Exceeded max queries per resolve ").append(this.maxAllowedQueries).append(' ');
            }
        }
        if (this.trace != null) {
            buf.append(':').append((CharSequence)this.trace);
        }
        UnknownHostException cause = new UnknownHostException(buf.toString());
        this.resolveCache.cache(this.hostname, cause, this.parent.ch.eventLoop());
        this.promise.tryFailure(cause);
    }

    protected abstract boolean finishResolve(Class<? extends InetAddress> var1, List<DnsCacheEntry> var2);

    static String decodeDomainName(ByteBuf in) {
        in.markReaderIndex();
        try {
            String string = DnsNameResolverContext.decodeName(in);
            return string;
        }
        catch (CorruptedFrameException e) {
            String string = null;
            return string;
        }
        finally {
            in.resetReaderIndex();
        }
    }

    private void followCname(InetSocketAddress nameServerAddr, String name, String cname) {
        if (this.traceEnabled) {
            if (this.trace == null) {
                this.trace = new StringBuilder(128);
            }
            this.trace.append(StringUtil.NEWLINE);
            this.trace.append("\tfrom ");
            this.trace.append(nameServerAddr);
            this.trace.append(": ");
            this.trace.append(name);
            this.trace.append(" CNAME ");
            this.trace.append(cname);
        }
        InetSocketAddress nextAddr = this.nameServerAddrs.next();
        this.query(nextAddr, new DefaultDnsQuestion(cname, DnsRecordType.A));
        this.query(nextAddr, new DefaultDnsQuestion(cname, DnsRecordType.AAAA));
    }

    private void addTrace(InetSocketAddress nameServerAddr, String msg) {
        assert (this.traceEnabled);
        if (this.trace == null) {
            this.trace = new StringBuilder(128);
        }
        this.trace.append(StringUtil.NEWLINE);
        this.trace.append("\tfrom ");
        this.trace.append(nameServerAddr);
        this.trace.append(": ");
        this.trace.append(msg);
    }

    private void addTrace(Throwable cause) {
        assert (this.traceEnabled);
        if (this.trace == null) {
            this.trace = new StringBuilder(128);
        }
        this.trace.append(StringUtil.NEWLINE);
        this.trace.append("Caused by: ");
        this.trace.append(cause);
    }

    public static String decodeName(ByteBuf in) {
        int position = -1;
        int checked = 0;
        int end = in.writerIndex();
        int readable = in.readableBytes();
        if (readable == 0) {
            return ".";
        }
        StringBuilder name = new StringBuilder(readable << 1);
        while (in.isReadable()) {
            boolean pointer;
            short len = in.readUnsignedByte();
            boolean bl = pointer = (len & 0xC0) == 192;
            if (pointer) {
                if (position == -1) {
                    position = in.readerIndex() + 1;
                }
                if (!in.isReadable()) {
                    throw new CorruptedFrameException("truncated pointer in a name");
                }
                int next = (len & 0x3F) << 8 | in.readUnsignedByte();
                if (next >= end) {
                    throw new CorruptedFrameException("name has an out-of-range pointer");
                }
                in.readerIndex(next);
                if ((checked += 2) < end) continue;
                throw new CorruptedFrameException("name contains a loop.");
            }
            if (len == 0) break;
            if (!in.isReadable(len)) {
                throw new CorruptedFrameException("truncated label in a name");
            }
            name.append(in.toString(in.readerIndex(), len, CharsetUtil.UTF_8)).append('.');
            in.skipBytes(len);
        }
        if (position != -1) {
            in.readerIndex(position);
        }
        if (name.length() == 0) {
            return ".";
        }
        if (name.charAt(name.length() - 1) != '.') {
            name.append('.');
        }
        return name.toString();
    }
}

