package org.immutables.value.processor.encode;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.*;

/**
 * EncMetadataMirror used to parse data of AnnotationMirror for original annotation {@code org.immutables.encode.EncodingMetadata}
 * during annotation processing. Interface is being described using {@link org.immutables.value.processor.encode.Mirrors.EncMetadata} annotation,
 * which should be structurally compatible to the annotation being modelled.
 * @see #find(Iterable)
 * @see #from(AnnotationMirror)
 */
@SuppressWarnings("all")
public class EncMetadataMirror implements Mirrors.EncMetadata {
  public static final String QUALIFIED_NAME = "org.immutables.encode.EncodingMetadata";
  public static final String MIRROR_QUALIFIED_NAME = "org.immutables.value.processor.encode.Mirrors.EncMetadata";

  public static String mirrorQualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String qualifiedName() {
    return QUALIFIED_NAME;
  }

  public static String simpleName() {
    return "EncodingMetadata";
  }

  public static boolean isPresent(Element annotatedElement) {
    for (AnnotationMirror mirror : annotatedElement.getAnnotationMirrors()) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Finds first annotation of this type on the element.
   * @param element annotated element
   * @return optional {@code EncMetadataMirror}, present if this annotation found
   */
  public static Optional<EncMetadataMirror> find(Element element) {
    return find(element.getAnnotationMirrors());
  }

  /**
   * Finds first annotation of this type in an iterable of annotation mirrors.
   * @param mirrors annotation mirrors
   * @return optional {@code EncMetadataMirror}, present if this annotation found
   */
  public static Optional<EncMetadataMirror> find(Iterable<? extends AnnotationMirror> mirrors) {
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      if (element.getQualifiedName().contentEquals(QUALIFIED_NAME)) {
        return Optional.of(new EncMetadataMirror(mirror));
      }
    }
    return Optional.absent();
  }

  /**
   * Converts iterable of annotation mirrors where all annotation are of this type. Otherwise it fails
   * @param mirrors of this annotation type.
   * @return list of converted {@code EncMetadataMirror}s
   */
  public static ImmutableList<EncMetadataMirror> fromAll(Iterable<? extends AnnotationMirror> mirrors) {
    ImmutableList.Builder<EncMetadataMirror> builder = ImmutableList.builder();
    for (AnnotationMirror mirror : mirrors) {
      TypeElement element = (TypeElement) mirror.getAnnotationType().asElement();
      Preconditions.checkState(element.getQualifiedName().contentEquals(QUALIFIED_NAME),
          "Supplied mirrors should all be of this annotation type");
      builder.add(new EncMetadataMirror(mirror));
    }
    return builder.build();
  }

  /**
   * Creates mirror with default values using annotation element (i.e. declaration, not usage).
   * @param element annotation type element
   * @return {@code EncMetadataMirror}
   */
  public static EncMetadataMirror from(TypeElement element) {
    return new EncMetadataMirror(element);
  }

  /**
   * Tries to convert annotation mirror to this annotation type.
   * @param mirror annotation mirror
   * @return optional {@code EncMetadataMirror}, present if mirror matched this annotation type
   */
  public static Optional<EncMetadataMirror> from(AnnotationMirror mirror) {
    return find(Collections.singleton(mirror));
  }

  private final AnnotationMirror annotationMirror;
  private final String name;
  private final String[] imports;
  private final String[] typeParams;
  private final AnnotationMirror[] elements;
  private final EncElementMirror[] elementsMirror;

  private EncMetadataMirror(TypeElement defaultAnnotationElement) {
    Preconditions.checkArgument(defaultAnnotationElement.getQualifiedName().contentEquals(QUALIFIED_NAME)
        || defaultAnnotationElement.getQualifiedName().contentEquals(MIRROR_QUALIFIED_NAME));
    this.annotationMirror = null;

    // TBD TODO BIG

    String name = null;
    String[] imports = null;
    String[] typeParams = null;
    AnnotationMirror[] elements = null;
    EncElementMirror[] elementsMirror = null;

    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(defaultAnnotationElement.getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("name".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @EncodingMetadata");
        }
        NameExtractor nameExtractor$ = new NameExtractor();
        annotationValue$.accept(nameExtractor$, null);

        name = nameExtractor$.get();
        continue;
      }
      if ("imports".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @EncodingMetadata");
        }
        ImportsExtractor importsExtractor$ = new ImportsExtractor();
        annotationValue$.accept(importsExtractor$, null);

        imports = importsExtractor$.get();
        continue;
      }
      if ("typeParams".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @EncodingMetadata");
        }
        TypeParamsExtractor typeParamsExtractor$ = new TypeParamsExtractor();
        annotationValue$.accept(typeParamsExtractor$, null);

        typeParams = typeParamsExtractor$.get();
        continue;
      }
      if ("elements".equals(name$)) {
        AnnotationValue annotationValue$ = attributeElement$.getDefaultValue();
        if (annotationValue$ == null) {
          throw new IllegalStateException("All annotation attributes should have default value to create mirror of @EncodingMetadata");
        }
        ElementsExtractor elementsExtractor$ = new ElementsExtractor();
        annotationValue$.accept(elementsExtractor$, null);

        elements = elementsExtractor$.get();
        elementsMirror = elementsExtractor$.mirror();
        continue;
      }
    }
    this.name = Preconditions.checkNotNull(name, "default attribute 'name'");
    this.imports = Preconditions.checkNotNull(imports, "default attribute 'imports'");
    this.typeParams = Preconditions.checkNotNull(typeParams, "default attribute 'typeParams'");
    this.elements = Preconditions.checkNotNull(elements, "default attribute 'elements'");
    this.elementsMirror = Preconditions.checkNotNull(elementsMirror, "default attribute 'elements'");
  }

  private EncMetadataMirror(AnnotationMirror annotationMirror) {
    this.annotationMirror = annotationMirror;

    String name = null;
    String[] imports = null;
    String[] typeParams = null;
    AnnotationMirror[] elements = null;
    EncElementMirror[] elementsMirror = null;

    Map<? extends ExecutableElement, ? extends AnnotationValue> attributeValues$ = annotationMirror.getElementValues();
    for (ExecutableElement attributeElement$
        : ElementFilter.methodsIn(annotationMirror.getAnnotationType().asElement().getEnclosedElements())) {
      String name$ = attributeElement$.getSimpleName().toString();
      if ("name".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'name' attribute of @EncodingMetadata");
        }
        NameExtractor nameExtractor$ = new NameExtractor();
        annotationValue$.accept(nameExtractor$, null);

        name = nameExtractor$.get();
        continue;
      }
      if ("imports".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'imports' attribute of @EncodingMetadata");
        }
        ImportsExtractor importsExtractor$ = new ImportsExtractor();
        annotationValue$.accept(importsExtractor$, null);

        imports = importsExtractor$.get();
        continue;
      }
      if ("typeParams".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'typeParams' attribute of @EncodingMetadata");
        }
        TypeParamsExtractor typeParamsExtractor$ = new TypeParamsExtractor();
        annotationValue$.accept(typeParamsExtractor$, null);

        typeParams = typeParamsExtractor$.get();
        continue;
      }
      if ("elements".equals(name$)) {
        AnnotationValue annotationValue$ = attributeValues$.get(attributeElement$);
        if (annotationValue$ == null) {
          annotationValue$ = attributeElement$.getDefaultValue();
        }
        if (annotationValue$ == null) {
          throw new IllegalStateException("Annotation mirror contains no value (neither default) for 'elements' attribute of @EncodingMetadata");
        }
        ElementsExtractor elementsExtractor$ = new ElementsExtractor();
        annotationValue$.accept(elementsExtractor$, null);

        elements = elementsExtractor$.get();
        elementsMirror = elementsExtractor$.mirror();
        continue;
      }
    }
    this.name = Preconditions.checkNotNull(name, "value for 'name'");
    this.imports = Preconditions.checkNotNull(imports, "value for 'imports'");
    this.typeParams = Preconditions.checkNotNull(typeParams, "value for 'typeParams'");
    this.elements = Preconditions.checkNotNull(elements, "value for 'elements'");
    this.elementsMirror = Preconditions.checkNotNull(elementsMirror, "value for 'elements'");
  }

  /**
   * @return value of attribute {@code name}
   */
  @Override
  public String name() {
    return name;
  }

  /**
   * @return value of attribute {@code imports}
   */
  @Override
  public String[] imports() {
    return imports.clone();
  }

  /**
   * @return value of attribute {@code typeParams}
   */
  @Override
  public String[] typeParams() {
    return typeParams.clone();
  }

  /**
   * @return annotation mirror for value of attribute {@code elements}
   */
  public AnnotationMirror[] elementsMirror() {
    return elements.clone();
  }

  /**
   * @return reified annotation mirror for {@link org.immutables.value.processor.encode.Mirrors.EncElement}
   */
  @Override
  public EncElementMirror[] elements() {
    return elementsMirror;
  }

  /**
   * @return underlying annotation mirror
   */
  public AnnotationMirror getAnnotationMirror() {
    Preconditions.checkState(annotationMirror != null, "this is default mirror without originating AnnotationMirror");
    return annotationMirror;
  }

  /**
   * @return {@code EncMetadata.class}
   */
  @Override
  public Class<? extends Annotation> annotationType() {
    return Mirrors.EncMetadata.class;
  }

  @Override
  public int hashCode() {
    int h = 0;
    h += 127 * "name".hashCode() ^ name.hashCode();
    h += 127 * "imports".hashCode() ^ Arrays.hashCode(imports);
    h += 127 * "typeParams".hashCode() ^ Arrays.hashCode(typeParams);
    h += 127 * "elements".hashCode() ^ Arrays.hashCode(elements);
    return h;
  }

  @Override
  public boolean equals(Object other) {
    if (other instanceof EncMetadataMirror) {
      EncMetadataMirror otherMirror = (EncMetadataMirror) other;
      return name.equals(otherMirror.name)
          && Arrays.equals(imports, otherMirror.imports)
          && Arrays.equals(typeParams, otherMirror.typeParams)
          && Arrays.equals(elements, otherMirror.elements);
    }
    return false;
  }

  @Override
  public String toString() {
    return "EncMetadataMirror:" + annotationMirror;
  }

  private static class NameExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String value;

    @Override
    public Void visitString(String value, Void p) {
      this.value = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      Preconditions.checkState(!array.isEmpty());
      array.get(0).accept(this, null);
      return null;
    }

    String get() {
      return value;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'name' in @" + QUALIFIED_NAME);
    }
  }

  private static class ImportsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String[] values;
    int position;

    @Override
    public Void visitString(String value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new String[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    String[] get() {
      return values;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'imports' in @" + QUALIFIED_NAME);
    }
  }

  private static class TypeParamsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    String[] values;
    int position;

    @Override
    public Void visitString(String value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new String[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    String[] get() {
      return values;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'typeParams' in @" + QUALIFIED_NAME);
    }
  }

  private static class ElementsExtractor extends SimpleAnnotationValueVisitor7<Void, Void> {
    AnnotationMirror[] values;
    int position;

    @Override
    public Void visitAnnotation(AnnotationMirror value, Void p) {
      this.values[position++] = value;
      return null;
    }

    @Override
    public Void visitArray(List<? extends AnnotationValue> array, Void p) {
      this.values = new AnnotationMirror[array.size()];
      Verify.verify(position == 0);
      for (AnnotationValue value : array) {
        value.accept(this, null);
      }
      return null;
    }

    AnnotationMirror[] get() {
      return values;
    }

    public EncElementMirror[] mirror() {
      EncElementMirror[] mirrors = new EncElementMirror[values.length];
      for (int i = 0; i < values.length; i++) {
        mirrors[i] = EncElementMirror.from(values[i]).get();
      }
      return mirrors;
    }

    @Override
    protected Void defaultAction(Object o, Void p) {
      throw new IllegalStateException("Incompatible annotation content of attribute 'elements' in @" + QUALIFIED_NAME);
    }
  }
}
