/*
 * JBoss, Home of Professional Open Source
 * Copyright 2014 Red Hat Inc. and/or its affiliates and other contributors
 * as indicated by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import org.jboss.aesh.readline.ActionDecoder;
import org.jboss.aesh.readline.KeyAction;
import org.jboss.aesh.readline.Prompt;
import org.jboss.aesh.readline.Readline;
import org.jboss.aesh.readline.ReadlineBuilder;
import org.jboss.aesh.readline.completion.CompleteOperation;
import org.jboss.aesh.readline.completion.Completion;
import org.jboss.aesh.terminal.Key;
import org.jboss.aesh.tty.terminal.TerminalConnection;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;

/**
 * Shell.java replacement using AEsh
 * Created by thauser on 8/20/14.
 */
public class Shell {

    private TerminalConnection connection;
    private Readline readline;
    private Prompt prompt;
    private List<Completion> completions;
    private static Shell SHELL;
    private boolean directoryField;
    private boolean disableFileCompletion;
    private volatile String input;
    private volatile boolean lineRead = false;

    private Shell() {
        connection = new TerminalConnection();
        readline = ReadlineBuilder.builder().enableHistory(false).build();
        completions = new ArrayList<>();
        completions.add(fileCompleter);
        prompt = new Prompt("[prompt]");
    }

    public static Shell getInstance() {
        if (SHELL == null) {
            SHELL = new Shell();
        }
        return SHELL;
    }

    public String getInput() {
        final String[] line = new String[1];
        readline.readline(connection, prompt, input -> {
            line[0] = input;
            connection.stopReading();
        });
        connection.startBlockingReader();
        return line[0];
    }


    /**
     * Get a folder location from the user.
     * Supports autocomplete features.
     * We trim off any trailing whitepsace.
     * The check for line.length() > 1, is an exception to leave the slash for the root directory.
     */
    public String getLocation(boolean directoryField) {
        setDirectoryField(directoryField);
        final String[] line = new String[1];
        readline.readline(connection, prompt, input -> {
            line[0] = input;
        }, completions);
        return line[0].replace("~", System.getProperty("user.home")).trim();
    }

    /**
     * Get a password from the user.
     * Currently the ConsoleReader does not support hiding the password being entered.
     */
    public String getPassword() {
        final String[] line = new String[1];
        readline.readline(connection, new Prompt("", '*'), input -> {
            line[0] = input;
            connection.stopReading();
        });
        connection.startBlockingReader();
        return line[0];
    }

    /**
     * Get a single character from the user
     *
     * @return A character that the user has entered.
     */
    public char getChar() {
        final KeyAction[] action = new KeyAction[1];
        ActionDecoder decoder = new ActionDecoder();
        CountDownLatch latch = new CountDownLatch(1);
        connection.setStdinHandler( ints -> {
            decoder.add(ints);
            action[0] = decoder.next();
            connection.stopReading();
            latch.countDown();
        });
        connection.startBlockingReader();

        try {
            latch.await();
            //this will only return printable chars
            //if you want to return everything, just:
            //return (char) action[0].getCodePointAt(0);
            if (action[0] instanceof Key) {
                if (((Key) action[0]).isPrintable())
                    return ((Key) action[0]).getAsChar();
                else {
                    return '\u0000';
                }
            }
            return (char) action[0].getCodePointAt(0);
        }
        catch (InterruptedException e) {
            return '\u0000';
        }
        finally {
            connection.setStdinHandler(null);
        }
    }

    private void setDirectoryField(boolean isDirectory) {
        this.directoryField = isDirectory;
    }

    private void setDisableFileCompletion(boolean disableFileCompletion) {
        this.disableFileCompletion = disableFileCompletion;
    }

    private boolean getDirectoryField() {
        return directoryField;
    }

    private boolean getDisableFileCompletion() {
        return disableFileCompletion;
    }

    Completion fileCompleter = new Completion() {
        @Override
        public void complete(CompleteOperation co) {

            if (getDisableFileCompletion())
                return;

            /*
            String originalEntry = co.getBuffer().trim();
            String subbedEntry = originalEntry;
            if (subbedEntry.startsWith("~")) {
                subbedEntry = subbedEntry.replaceFirst("~", System.getProperty("user.home"));
            }
            String rest = getLastPathSection(subbedEntry);
            File currFile = new File(subbedEntry);

            if (!subbedEntry.isEmpty() && (currFile.isDirectory() || currFile.getParent() != null)) {

                File[] filesInDir = null;
                PrefixFileFilter fileFilter = new PrefixFileFilter(rest, getDirectoryField());

                // handle case of root directory, ie "/"
                if (currFile.getParentFile() == null && currFile.isDirectory()) {

                    if (System.getProperty("os.name").startsWith("Windows")) {
                        Pattern root = Pattern.compile("^[A-Z]:$");
                        Matcher rootMatcher = root.matcher(currFile.getPath());
                        if (rootMatcher.matches()) {
                            co.addCompletionCandidate("\\");
                        } else {
                            filesInDir = currFile.listFiles(fileFilter);
                        }
                    } else {
                        filesInDir = currFile.listFiles(fileFilter);
                    }
                }
                // handle case of paths that have a parent dir but aren't a dir or file them selves, "/home/username/not_fini"
                else if (!currFile.exists() && currFile.getParentFile() != null) {
                    if (System.getProperty("os.name").startsWith("Windows") && !currFile.getPath().contains(System.getProperty("file.separator"))) {
                        filesInDir = null;
                    } else {
                        filesInDir = currFile.getParentFile().listFiles(fileFilter);
                    }
                }
                // if it is either a file or directory and not the root dir
                else if (currFile.exists()) {
                    // This means that its a file or dir and the only one that is possible at this point, so look for things in this or file now
                    if (rest.length() > 0) {
                        filesInDir = currFile.getParentFile().listFiles(fileFilter);
                    }
                    // Means that it ended in slash meaning a new dir
                    else {
                        filesInDir = currFile.listFiles(fileFilter);
                    }
                }

                if (filesInDir != null) {
                    for (File f : filesInDir) {
                        String candidate = f.getAbsolutePath();
                        candidate = getLastPathSection(candidate);
                        if (f.isDirectory()) {
                            candidate = candidate + System.getProperty("file.separator");
                        }
                        co.addCompletionCandidate(candidate);
                    }
                }
            }

            co.setOffset(originalEntry.length() - rest.length());
            co.doAppendSeparator(false);
            */
        }
    };

    private String getLastPathSection(String path) {
        if (path.endsWith(System.getProperty("file.separator"))) {
            return "";
        }
        String fileSeparator = System.getProperty("file.separator");
        // Because \\ is not a valid regex
        if (fileSeparator.equals("\\")) {
            fileSeparator = fileSeparator.concat(fileSeparator);
        }
        String[] pathSegments = path.split(fileSeparator);
        String rest = "";
        if (pathSegments.length > 1) {
            rest = pathSegments[pathSegments.length - 1];
        }
        return rest;
    }

    public static void main(String[] args) {
        Shell shell = Shell.getInstance();
        System.out.println("trying to get a line:");
        String line = shell.getInput();
        System.out.println("we got: "+line+", lets get a password..");
        String pass = shell.getPassword();
        System.out.println("password: "+pass+", lets get a char: ");
        char in = shell.getChar();
        System.out.println("got char: "+in);
    }
}