/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.aspects.security;

import org.jboss.aop.joinpoint.Invocation;
import org.jboss.security.AuthenticationManager;
import org.jboss.security.RealmMapping;
import org.jboss.security.RunAsIdentity;

/**
 * An interceptor that enforces the run-as identity declared by a bean.
 * 
 * @author <a href="mailto:Scott.Stark@jboss.org">Scott Stark</a>.
 * @author <a href="mailto:dain@daingroup.com">Dain Sundstrom</a>.
 * @version $Revision: 65273 $
 */
public class RunAsSecurityInterceptor implements org.jboss.aop.advice.Interceptor
{
   protected AuthenticationManager securityManager;

   protected RealmMapping realmMapping;

   /**
    * <p>
    * Creates an instance of {@code RunAsSecurityInterceptor}.
    * </p>
    */
   public RunAsSecurityInterceptor()
   {
   }

   /**
    * <p>
    * Creates an instance of {@code RunAsSecurityInterceptor} using the specified {@code AuthenticationManager}
    * and {@code RealmMapping} implementations.
    * </p>
    * 
    * @param manager the {@code AuthenticationManager} instance to be used when the caller hasn't been previously
    *            authenticated.
    * @param realmMapping the {@code RealmMapping} instance to be used to determine if the caller has or has not the
    *            required roles.
    * @deprecated use {@code #RunAsSecurityInterceptor()} instead.
    */
   @Deprecated
   public RunAsSecurityInterceptor(AuthenticationManager manager, RealmMapping realmMapping)
   {
      this.securityManager = manager;
      this.realmMapping = realmMapping;
   }

   /*
    * (non-Javadoc)
    * @see org.jboss.aop.advice.Interceptor#getName()
    */
   public String getName()
   {
      return "RunAsSecurityInterceptor";
   }

   /**
    * <p>
    * Obtains the run-as identity to be used when running the method represented by the {@code Invocation}.
    * </p>
    * 
    * @param invocation the object that contains the metadata of the method being called.
    * @return the {@code RunAsIdentity} to be used, or {@code null} if no run-as identity can be found.
    */
   protected RunAsIdentity getRunAsIdentity(Invocation invocation)
   {
      RunAsIdentity identity = (RunAsIdentity) invocation.getMetaData("security", "run-as");
      if (identity == null)
         identity = getAnnotationRunAsIdentity(invocation);
      return identity;
   }

   /**
    * <p>
    * Obtains the run-as identity that have been specified through annotations.
    * </p>
    * 
    * @param invocation the object that contains the metadata of the method being called.
    * @return the {@code RunAsIdentity} to be used, or {@code null} if no run-as identity has been specified..
    */
   protected RunAsIdentity getAnnotationRunAsIdentity(Invocation invocation)
   {
      RunAs runAs = (RunAs) invocation.resolveAnnotation(RunAs.class);
      if (runAs == null)
      {
         runAs = (RunAs) invocation.resolveClassAnnotation(RunAs.class);
      }
      if (runAs == null)
         return null;
      RunAsIdentity runAsRole = new RunAsIdentity(runAs.value(), null);
      return runAsRole;
   }

   /*
    * (non-Javadoc)
    * @see org.jboss.aop.advice.Interceptor#invoke(org.jboss.aop.joinpoint.Invocation)
    */
   public Object invoke(org.jboss.aop.joinpoint.Invocation invocation) throws Throwable
   {
      RunAsIdentity runAsRole = getRunAsIdentity(invocation);
      // If a run-as role was specified, push it so that any calls made
      // by this bean will have the runAsRole available for declarative
      // security checks.
      if (runAsRole != null)
      {
         SecurityActions.pushRunAsIdentity(runAsRole);
      }

      try
      {
         return invocation.invokeNext();
      }
      finally
      {
         if (runAsRole != null)
         {
            SecurityActions.popRunAsIdentity();
         }
      }
   }
}
