/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.server;

import javax.ws.rs.*;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Context;
import javax.servlet.http.HttpServletRequest;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.jboss.bpm.console.client.model.ProcessDefinitionRefWrapper;
import org.jboss.bpm.console.client.model.ProcessInstanceRef;
import org.jboss.bpm.console.client.model.ProcessInstanceRefWrapper;
import org.jboss.bpm.console.server.gson.GsonFactory;
import org.jboss.bpm.console.server.integration.ManagementFactory;
import org.jboss.bpm.console.server.integration.ProcessManagement;

import com.google.gson.Gson;

import java.util.Iterator;
import java.util.List;
import java.io.InputStream;

/**
 * REST server module for accessing process related data.
 *
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
@Path("process")
public class ProcessMgmtFacade
{
  private static final Log log = LogFactory.getLog(ProcessMgmtFacade.class);
  private ProcessManagement processManagement;

  private ProcessManagement getProcessManagement()
  {
    if(null==this.processManagement)
    {
      ManagementFactory factory = ManagementFactory.newInstance();
      this.processManagement = factory.createProcessManagement();
      log.debug("Using ManagementFactory impl:" + factory.getClass().getName());
    }

    return this.processManagement;
  }

  @GET
  @Path("definitions")
  @Produces("application/json")
  public Response getDefinitionsJSON()
  {
    ProcessDefinitionRefWrapper wrapper =
        new ProcessDefinitionRefWrapper(getProcessManagement().getProcessDefinitions());
    return createJsonResponse(wrapper);
  }

  @POST
  @Path("definitions/{id}/remove")
  @Produces("application/json")
  public Response removeDefinitionsJSON(
      @PathParam("id")
      String definitionId
  )
  {
    ProcessDefinitionRefWrapper wrapper =
        new ProcessDefinitionRefWrapper( getProcessManagement().removeProcessDefinition(definitionId));
    return createJsonResponse(wrapper);
  }

  @GET
  @Path("definitions/{id}/instances")
  @Produces("application/json")
  public Response getInstancesJSON(
      @PathParam("id")
      String definitionId
  )
  {
    ProcessInstanceRefWrapper wrapper =
        new ProcessInstanceRefWrapper(getProcessManagement().getProcessInstances(definitionId));
    return createJsonResponse(wrapper);
  }

  @POST
  @Path("definitions/{id}/instances/new")
  @Produces("application/json")
  public Response newInstance(
      @PathParam("id")
      String definitionId)
  {

    ProcessInstanceRef instance = null;
    try
    {
      instance = getProcessManagement().newInstance(definitionId);
      return createJsonResponse(instance);
    }
    catch (Throwable t)
    {      
      throw new WebApplicationException(t, 500);
    }

  }

  @POST
  @Path("instances/{id}/state/{next}")
  @Produces("application/json")
  public Response changeState(
      @PathParam("id")
      String executionId,
      @PathParam("next")
      String next)
  {
    ProcessInstanceRef.STATE state = ProcessInstanceRef.STATE.valueOf(next);
    log.debug("Change instance (ID "+executionId+") to state " +state);
    getProcessManagement().setProcessState(executionId, state);
    return Response.ok().type("application/json").build();
  }

  @POST
  @Path("tokens/{id}/transition")
  @Produces("application/json")
  public Response signalExecution(
      @PathParam("id")
      String id,
      @QueryParam("signal")
      String signalName)
  {
    log.debug("Signal token " + id + " -> " + signalName);

    if ("default transition".equals(signalName))
      signalName = null;

    getProcessManagement().signalExecution(id, signalName);
    return Response.ok().type("application/json").build();
  }

  @POST
  @Path("tokens/{id}/transition/default")
  @Produces("application/json")
  public Response signalExecutionDefault(
      @PathParam("id")
      String id)
  {
    log.debug("Signal token " + id);

    getProcessManagement().signalExecution(id, null);
    return Response.ok().type("application/json").build();
  }

  @POST
  @Path("definitions/new")
  @Produces("application/json")
  @Consumes("multipart/form-data")
  public Response postNewDefinition(
      @Context
      HttpServletRequest request
  )
  {
    try
    {
      FileItemFactory factory = new DiskFileItemFactory();
      ServletFileUpload upload = new ServletFileUpload(factory);
      List items = upload.parseRequest(request);

      Iterator iter = items.iterator();
      while (iter.hasNext())
      {
        FileItem item = (FileItem) iter.next();

        if (item.isFormField())
        {
          // ignore
          System.out.println("Caught form field on file upload: " + item.getName());
        }
        else
        {
          String fieldName = item.getFieldName();
          String fileName = item.getName();
          String contentType = item.getContentType();
          boolean isInMemory = item.isInMemory();
          long sizeInBytes = item.getSize();

          // Process a file upload in memory
          //byte[] data = item.get();

          // Process stream
          InputStream uploadedStream = item.getInputStream();
          getProcessManagement().deploy(fileName, contentType, uploadedStream);
          uploadedStream.close();


        }

      }
    } catch (Exception e)
    {
      return Response.serverError().build();
    }

    return Response.ok().build();
  }


  private Response createJsonResponse(Object wrapper)
  {
    Gson gson = GsonFactory.createInstance();
    String json = gson.toJson(wrapper);
    return Response.ok(json).type("application/json").build();
  }
}
