/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.server;

import com.google.gson.Gson;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.console.client.model.TaskRefWrapper;
import org.jboss.bpm.console.client.model.TaskRef;
import org.jboss.bpm.console.server.gson.GsonFactory;
import org.jboss.bpm.console.server.integration.ManagementFactory;
import org.jboss.bpm.console.server.integration.TaskManagement;
import org.jboss.bpm.console.server.plugin.TaskDispatcherPlugin;
import org.jboss.bpm.console.server.plugin.PluginMgr;

import javax.ws.rs.*;
import javax.ws.rs.core.Response;
import java.util.List;
import java.net.URL;

/**
 * REST server module for accessing task related data.
 *
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
@Path("tasks")
public class TaskListFacade
{
  private static final Log log = LogFactory.getLog(TaskMgmtFacade.class);

  private TaskManagement taskManagement;
  private TaskDispatcherPlugin formPlugin;

  /**
   * Lazy load the {@link org.jboss.bpm.console.server.integration.TaskManagement}
   */
  private TaskManagement getTaskManagement()
  {
    if(null==this.taskManagement)
    {
      ManagementFactory factory = ManagementFactory.newInstance();
      this.taskManagement = factory.createTaskManagement();
      log.debug("Using ManagementFactory impl:" + factory.getClass().getName());
    }

    return this.taskManagement;
  }

  /**
   * Lazy load the {@link org.jboss.bpm.console.server.plugin.TaskDispatcherPlugin}.
   * Can be null if the plugin is not available.
   */
  private TaskDispatcherPlugin getTaskDispatcherPlugin()
  {
    if(null==this.formPlugin)
    {
      this.formPlugin = PluginMgr.load(TaskDispatcherPlugin.class);
    }

    return this.formPlugin;
  }

  @GET
  @Path("{idRef}")
  @Produces("application/json")
  public Response getTasksForIdRef(
      @PathParam("idRef")
      String idRef
  )
  {
    List<TaskRef> refs = getTaskManagement().getTasksForIdentity(idRef);

    // decorate task form URL if plugin available
    TaskDispatcherPlugin formPlugin = getTaskDispatcherPlugin();
    if(formPlugin!=null)
    {
      for(TaskRef task : refs)
      {
        URL taskFormURL = formPlugin.getDispatchUrl(task.getId());
        if(taskFormURL!=null)
        {
          task.setUrl(taskFormURL.toExternalForm());
        }
      }
    }
    
    TaskRefWrapper wrapper = new TaskRefWrapper(refs);
    return createJsonResponse(wrapper);
  }

  @GET
  @Path("{idRef}/participation/{type}")
  @Produces("application/json")
  public Response getTasksForIdRefAndParticipation(
      @PathParam("idRef")
      String idRef,
      @PathParam("type")
      String participationType
  )
  {
    TaskRefWrapper wrapper =
        new TaskRefWrapper(getTaskManagement().getTasksForIdentity(idRef, participationType));
    return createJsonResponse(wrapper);
  }

  private Response createJsonResponse(Object wrapper)
  {
    Gson gson = GsonFactory.createInstance();
    String json = gson.toJson(wrapper);
    return Response.ok(json).type("application/json").build();
  }
}
