/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.server;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.console.server.integration.ManagementFactory;
import org.jboss.bpm.console.server.integration.TaskManagement;
import org.jboss.bpm.console.server.plugin.PluginMgr;
import org.jboss.bpm.console.server.plugin.TaskDispatcherPlugin;
import org.jboss.resteasy.plugins.providers.multipart.InputPart;
import org.jboss.resteasy.plugins.providers.multipart.MultipartFormDataInput;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.io.InputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.ByteArrayInputStream;

/**
 * REST server module for accessing task related data.
 *
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
@Path("task")
public class TaskMgmtFacade
{
  private static final Log log = LogFactory.getLog(TaskMgmtFacade.class);

  private TaskManagement taskManagement;

  private TaskDispatcherPlugin formPlugin;

  /**
   * Lazy load the {@link org.jboss.bpm.console.server.integration.TaskManagement}
   */
  private TaskManagement getTaskManagement()
  {
    if(null==this.taskManagement)
    {
      ManagementFactory factory = ManagementFactory.newInstance();
      this.taskManagement = factory.createTaskManagement();
      log.debug("Using ManagementFactory impl:" + factory.getClass().getName());
    }

    return this.taskManagement;
  }

  /**
   * Lazy load the {@link org.jboss.bpm.console.server.integration.TaskManagement}
   */
  private TaskDispatcherPlugin getTaskDispatcherPlugin()
  {
    if(null==this.formPlugin)
    {
      this.formPlugin = PluginMgr.load(TaskDispatcherPlugin.class);
      log.debug("Using TaskDispatcherPlugin impl:" + this.formPlugin);
    }

    return this.formPlugin;
  }

  @POST
  @Path("{taskId}/assign/{ifRef}")
  @Produces("application/json")
  public Response assignTask(
      @PathParam("taskId")
      long taskId,
      @PathParam("ifRef")
      String idRef
  )
  {
    log.debug("Assign task " + taskId + " to '" + idRef +"'");
    getTaskManagement().assignTask(taskId, idRef);
    return Response.ok().build();
  }

  @POST
  @Path("{taskId}/release")
  @Produces("application/json")
  public Response releaseTask(
      @PathParam("taskId")
      long taskId
  )
  {
    log.debug("Release task " + taskId);
    getTaskManagement().assignTask(taskId, null);
    return Response.ok().build();
  }

  @POST
  @Path("{taskId}/close")
  @Produces("application/json")
  public Response closeTask(
      @PathParam("taskId")
      long taskId
  )
  {
    log.debug("Close task " + taskId );
    getTaskManagement().completeTask(taskId, null);
    return Response.ok().build();
  }

  @POST
  @Path("{taskId}/close/{outcome}")
  @Produces("application/json")
  public Response closeTaskWithSignal(
      @PathParam("taskId")
      long taskId,
      @QueryParam("outcome")
      String outcome
  )
  {
    log.debug("Close task " + taskId + " outcome " + outcome);
    getTaskManagement().completeTask(taskId, outcome, null);
    return Response.ok().build();
  }

  @GET
  @Path("{taskId}/render")
  @Produces("text/html")
  public Response renderUI(
      @PathParam("taskId")
      long taskId
  )
  {
    DataHandler dh = getTaskDispatcherPlugin().provideTaskUI(taskId);
    if(null==dh)
    {
      throw new RuntimeException("No UI associated with task ID " + taskId);
    }

    return Response.ok(dh.getDataSource()).type("text/html").build();
  }

  @POST
  @Path("{taskId}/process")
  @Produces("text/html")
  @Consumes("multipart/form-data")
  public Response closeWithUI(
      @PathParam("taskId")
      long taskId,
      MultipartFormDataInput payload
  )
  {
    Map<String,Object> processVars = new HashMap<String,Object>();

    Map<String,InputPart> formData = payload.getFormData();
    Iterator<String> partNames = formData.keySet().iterator();

    String outcome = null;
    
    while(partNames.hasNext())
    {
      final String partName = partNames.next();
      final InputPart part = formData.get(partName);
      final MediaType mediaType = part.getMediaType();

      if(partName.equals("outcome"))
      {
        outcome = part.getBodyAsString();      
      }
      else if(MediaType.TEXT_PLAIN_TYPE.equals(mediaType))
      {
        // RFC2045: Each part has an optional "Content-Type" header
        // that defaults to "text/plain".
        // Can go into process without conversion
        processVars.put(partName, part.getBodyAsString()); 
      }
      else
      {
        // anything else turns into a DataHandler
        final byte[] data = part.getBodyAsString().getBytes();
        DataHandler dh = new DataHandler(
            new DataSource()
            {
              public InputStream getInputStream() throws IOException
              {
                return new ByteArrayInputStream(data);
              }

              public OutputStream getOutputStream() throws IOException
              {
                throw new RuntimeException("This is a readonly DataHandler"); 
              }

              public String getContentType()
              {
                return mediaType.getType();
              }

              public String getName()
              {
                return partName;
              }
            }
        );

        processVars.put(partName, dh);        
      }
    }

    // trigger execution
    TaskManagement tm = getTaskManagement();
    
    if(null==outcome)
      tm.completeTask(taskId, processVars);
    else
      tm.completeTask(taskId, outcome, processVars);


    return Response.ok("Successfully processed task UI").build();
  }
}
