/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.server;

import com.google.gson.Gson;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.console.client.model.TaskRefWrapper;
import org.jboss.bpm.console.server.gson.GsonFactory;
import org.jboss.bpm.console.server.integration.ManagementFactory;
import org.jboss.bpm.console.server.integration.TaskManagement;

import javax.ws.rs.*;
import javax.ws.rs.core.Response;

/**
 * REST server module for accessing task related data.
 *
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
@Path("tasks")
public class TaskMgmtFacade
{
   private static final Log log = LogFactory.getLog(TaskMgmtFacade.class);

   private TaskManagement taskManagement;

   /**
    * Lazy load the {@link org.jboss.bpm.console.server.integration.ProcessManagement}
    */
   private TaskManagement getTaskManagement()
   {
      if(null==this.taskManagement)
      {
         ManagementFactory factory = ManagementFactory.newInstance();
         this.taskManagement = factory.createTaskManagement();
         log.debug("Using ManagementFactory impl:" + factory.getClass().getName());
      }

      return this.taskManagement;
   }

   @GET
   @Path("actor/{actorName}")
   @Produces("application/json")
   public Response getTasksForActor(
         @PathParam("actorName")
         String actorName
   )
   {
      TaskRefWrapper wrapper =
            new TaskRefWrapper(getTaskManagement().getTasksByActor(actorName));
      return createJsonResponse(wrapper);
   }

   @POST
   @Path("{taskId}/assignment/{actorId}")
   @Produces("application/json")
   public Response assignTask(
         @PathParam("taskId")
         long taskId,
         @PathParam("actorId")
         String actorId
   )
   {
      log.debug("Reassign task " + taskId + " to '" + actorId +"'");
      getTaskManagement().reassignTask(taskId, actorId);
      return Response.ok().build();
   }

   @POST
   @Path("{taskId}/assignment")
   @Produces("application/json")
   public Response releaseTask(
         @PathParam("taskId")
         long taskId       
   )
   {
      log.debug("Release task " + taskId);
      getTaskManagement().reassignTask(taskId, null);
      return Response.ok().build();
   }

   @POST
   @Path("{taskId}/close/transition/default")
   @Produces("application/json")
   public Response closeTask(
         @PathParam("taskId")
         long taskId
   )
   {
      return closeTaskWithSignal(taskId, null);
   }

   @POST
   @Path("{taskId}/close/transition")
   @Produces("application/json")
   public Response closeTaskWithSignal(
         @PathParam("taskId")
         long taskId,
         @QueryParam("signal")
         String signalName
   )
   {
      log.debug("Close task " + taskId);
      getTaskManagement().closeTask(taskId, signalName);
      return Response.ok().build();
   }

   private Response createJsonResponse(Object wrapper)
   {
      Gson gson = GsonFactory.createInstance();
      String json = gson.toJson(wrapper);
      return Response.ok(json).type("application/json").build();
   }
}
