/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.process;

import com.google.gwt.user.client.ui.ClickListener;
import com.google.gwt.user.client.ui.Widget;
import com.google.gwt.core.client.GWT;
import com.mvc4g.client.Controller;
import com.mvc4g.client.ViewInterface;
import com.mvc4g.client.Event;
import org.gwt.mosaic.ui.client.ListBox;
import org.gwt.mosaic.ui.client.ToolBar;
import org.gwt.mosaic.ui.client.ToolButton;
import org.gwt.mosaic.ui.client.MessageBox;
import org.gwt.mosaic.ui.client.layout.BoxLayout;
import org.gwt.mosaic.ui.client.layout.BoxLayoutData;
import org.gwt.mosaic.ui.client.layout.LayoutPanel;
import org.gwt.mosaic.ui.client.list.DefaultListModel;
import org.jboss.bpm.console.client.model.ProcessDefinitionRef;
import org.jboss.bpm.console.client.model.ProcessInstanceRef;
import org.jboss.bpm.console.client.LazyPanel;
import org.jboss.bpm.console.client.icons.ConsoleIconBundle;
import org.jboss.bpm.console.client.common.AbstractView;

import java.util.List;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
class InstanceListView extends AbstractView
{
  public final static String ID = InstanceListView.class.getName();

  private Controller controller;

  private LayoutPanel instanceList = null;

  private ListBox<ProcessInstanceRef> listBox;

  private ProcessDefinitionRef currentDefinition;

  private boolean isInitialized;

  private List<ProcessInstanceRef> cachedInstances = null;

  public InstanceListView()
  {
    super();
    ConsoleIconBundle icons = GWT.create(ConsoleIconBundle.class);
    setTitle("Process Instances");
    setIcon(icons.instanceIcon());
  }

  public boolean isInitialized()
  {
    return isInitialized;
  }

  public void initialize()
  {
    if(!isInitialized)
    {
      instanceList = new LayoutPanel( new BoxLayout(BoxLayout.Orientation.VERTICAL));
      instanceList.setPadding(0);
      instanceList.setWidgetSpacing(0);

      listBox =
          new ListBox<ProcessInstanceRef>(
              new String[] {
                  "Instance ID", "State", "Start Date"}
          );

      listBox.setCellRenderer(new ListBox.CellRenderer<ProcessInstanceRef>() {
        public void renderCell(ListBox<ProcessInstanceRef> listBox, int row, int column,
                               ProcessInstanceRef item) {
          switch (column) {
            case 0:
              listBox.setText(row, column, item.getId());
              break;
            case 1:
              listBox.setText(row, column, item.getState().toString());
              break;
            case 2:
              listBox.setText(row, column, item.getStartDate().toString());
              break;
            default:
              throw new RuntimeException("Unexpected column size");
          }
        }
      });

      // toolbar
      final LayoutPanel toolBox = new LayoutPanel();
      toolBox.setPadding(0);
      toolBox.setWidgetSpacing(5);      

      final ToolBar toolBar = new ToolBar();
      toolBar.add(
          new ToolButton("Refresh", new ClickListener() {
            public void onClick(Widget sender) {
              controller.handleEvent(
                  new Event(
                      LoadInstancesAction.ID,
                      getCurrentDefinition()
                  )
              );
            }
          }
          )
      );

      toolBar.addSeparator();

      toolBar.add(
          new ToolButton("Start new instance", new ClickListener()
          {
            public void onClick(Widget sender)
            {
              controller.handleEvent(
                  new Event(
                      StartNewInstanceAction.ID,
                      getCurrentDefinition()
                  )
              );
            }
          }
          )
      );

      toolBar.addSeparator();

      toolBar.add(
          new ToolButton("Cancel execution", new ClickListener()
          {
            public void onClick(Widget sender)
            {
              ProcessInstanceRef selection = getSelection();
              if(selection!=null)
              {
                selection.setState(ProcessInstanceRef.STATE.ENDED);

                controller.handleEvent(
                    new Event(
                        StateChangeAction.ID,
                        selection
                    )
                );
              }
              else
              {
                MessageBox.alert("Missing selection", "Please select an instance");
              }
            }
          }
          )
      );

      toolBox.add(toolBar, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));

      instanceList.add(toolBox, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));
      instanceList.add(listBox, new BoxLayoutData(BoxLayoutData.FillStyle.BOTH));

      // cached data?
      if(this.cachedInstances!=null)
        bindData(this.cachedInstances);

      this.add(instanceList);

      isInitialized = true;

    }
  }

  public ProcessInstanceRef getSelection()
  {
    ProcessInstanceRef selection = null;
    if(listBox.getSelectedIndex()!=-1)
      selection = listBox.getItem( listBox.getSelectedIndex());
    return selection;
  }

  public ProcessDefinitionRef getCurrentDefinition()
  {
    return this.currentDefinition;
  }

  public void setController(Controller controller)
  {
    this.controller = controller;
  }


  public void update(final ProcessDefinitionRef def, List<ProcessInstanceRef> instances)
  {
    this.currentDefinition = def;
    this.cachedInstances = instances;

    if(isInitialized())
    {
      bindData(instances);
    }   
  }

  private void bindData(List<ProcessInstanceRef> instances)
  {
    final DefaultListModel<ProcessInstanceRef> model =
        (DefaultListModel<ProcessInstanceRef>) listBox.getModel();
    model.clear();

    for(ProcessInstanceRef inst : instances)
    {
      model.add(inst);
    }

    // layout again
    this.invalidate();
  }
}
