/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client;

import com.google.gwt.core.client.GWT;
import com.google.gwt.http.client.*;
import com.google.gwt.user.client.*;
import com.google.gwt.user.client.ui.*;
import com.mvc4g.client.Controller;
import com.mvc4g.client.ViewInterface;
import org.gwt.mosaic.ui.client.WindowPanel;
import org.gwt.mosaic.ui.client.InfoPanel;
import org.gwt.mosaic.ui.client.MessageBox;
import org.gwt.mosaic.ui.client.LayoutPopupPanel;
import org.gwt.mosaic.ui.client.layout.*;
import org.jboss.bpm.console.client.util.ConsoleLog;
import org.jboss.bpm.console.client.common.HeaderLabel;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class LoginView implements ViewInterface
{
  public final static String NAME = "loginView";

  private ConsoleConfig config;
  private Authentication auth;

  private WindowPanel window = null;
  private TextBox usernameInput;
  private PasswordTextBox passwordInput;

  private Controller controller;

  public final static String[] KNOWN_ROLES = {"admin", "manager", "user"};

  public LoginView()
  {

    // setup base urls
    String proxyUrl = null;
    if (!GWT.isScript())
    {
      proxyUrl = GWT.getModuleBaseURL() + "xhp";
    }

    config = new ConsoleConfig(proxyUrl);
    ConsoleLog.debug("Console server: " + config.getConsoleServerUrl());

    URLBuilder.configureInstance(config);
  }

  public void setController(Controller controller)
  {
    this.controller = controller;
  }

  public void display()
  {
    // force session invalidation, required to catch browser reload
    Authentication.logout(config);

    // start new session
    requestSessionID();
  }

  private void requestSessionID()
  {
    RequestBuilder rb = new RequestBuilder(
        RequestBuilder.GET,
        config.getConsoleServerUrl()+"/rs/identity/sid"
    );

    try
    {
      rb.sendRequest(null, new RequestCallback()
      {

        public void onResponseReceived(Request request, Response response)
        {
          ConsoleLog.debug("SID: "+ response.getText());
          ConsoleLog.debug("Cookies: "+ Cookies.getCookieNames());
          final String sid = response.getText();

          auth = new Authentication(
              config,
              sid,
              URLBuilder.getInstance().getUserInRoleURL(KNOWN_ROLES)
          );

          auth.setCallback(
              new Authentication.AuthCallback()
              {

                public void onLoginSuccess(Request request, Response response)
                {
                  // clear the form
                  usernameInput.setText("");
                  passwordInput.setText("");

                  // display main console
                  window.hide();

                  // assemble main layout
                  DeferredCommand.addCommand(
                      new Command()
                      {
                        public void execute()
                        {
                          // move the loading div to foreground
                          DOM.getElementById("splash").getStyle().setProperty("z-index", "1000");
                          DOM.getElementById("ui_loading").getStyle().setProperty("visibility", "visible");

                          // launch workspace
                          new MainLayout(controller, auth, config);
                        }
                      });

                  window = null;
                }

                public void onLoginFailed(Request request, Throwable t)
                {
                  // auth failed
                  MessageBox.error("Login failed", t.getMessage());
                  ConsoleLog.error("Login failed", t);
                }
              }
          );


          createLayoutWindowPanel();
          window.center();

          // focus
          usernameInput.setFocus(true);

        }

        public void onError(Request request, Throwable t)
        {
          ConsoleLog.error("Failed to initiate session", t);
        }
      });
    }
    catch (RequestException e)
    {
      ConsoleLog.error("Request error", e);
    }
  }

  /**
   * The 'layout' window panel.
   */
  private void createLayoutWindowPanel() {
    window = new WindowPanel(config.getProfileName());
    window.setAnimationEnabled(true);
    window.setSize("320px", "180px");

    LayoutPanel panel = new LayoutPanel();

    createLayoutContent(panel);
    window.setWidget(panel);

    window.addWindowCloseListener(new WindowCloseListener() {
      public void onWindowClosed() {
        window = null;
      }

      public String onWindowClosing() {
        return null;
      }
    });
  }

  /**
   * Create content for layout.
   */
  private void createLayoutContent(LayoutPanel layoutPanel) {

    layoutPanel.setLayout(new BorderLayout());
    layoutPanel.setPadding(5);


    Widget form = createForm();

    final Button submit = new Button("Submit");
    submit.addClickListener(new ClickListener()
    {

      public void onClick(Widget widget)
      {
        engageLogin();
      }
    });


    HTML html = new HTML("Version: " + Version.VERSION);
    html.setStyleName("bpm-login-info");

    LayoutPanel southContainer = new LayoutPanel(new BoxLayout(BoxLayout.Orientation.VERTICAL));
    southContainer.add(submit, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));
    southContainer.add(html);

    layoutPanel.add(form, new BorderLayoutData(BorderLayout.Region.CENTER));
    layoutPanel.add(southContainer, new BorderLayoutData(BorderLayout.Region.SOUTH));

  }

  private void engageLogin()
  {
    requestProtectedResource();
  }

  /**
   * The j_security_check URL is a kind of temporary resource that only exists
   * if tomcat decided that the login page should be shown.
   */
  private void requestProtectedResource()
  {
    RequestBuilder rb = new RequestBuilder(
        RequestBuilder.GET,
        config.getConsoleServerUrl()+"/rs/identity/secure/sid"
    );

    try
    {
      rb.sendRequest(null, new RequestCallback()
      {

        public void onResponseReceived(Request request, Response response)
        {
          ConsoleLog.debug("requestProtectedResource() HTTP "+response.getStatusCode());
          auth.login( getUsername(), getPassword() );
        }

        public void onError(Request request, Throwable t)
        {
          ConsoleLog.error("Failed to request protected resource", t);
        }
      });
    }
    catch (RequestException e)
    {
      ConsoleLog.error("Request error", e);
    }
  }

  private String getUsername()
  {
    return usernameInput.getText();
  }

  private String getPassword()
  {
    return passwordInput.getText();
  }

  private Widget createForm()
  {
    LayoutPanel p = new LayoutPanel(new BoxLayout(BoxLayout.Orientation.VERTICAL));
    HeaderLabel label = new HeaderLabel("Login", true);    
    p.add(label, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));

    usernameInput = new TextBox();
    passwordInput = new PasswordTextBox();

    Grid grid = new Grid(2,2);
    grid.setWidget(0,0, new Label("Username:"));
    grid.setWidget(0,1, usernameInput);

    grid.setWidget(1,0, new Label("Password:"));
    grid.setWidget(1,1, passwordInput);

    /*DisclosurePanel msg = new DisclosurePanel("Contact Address");

    grid.setWidget(2,0, new Label(""));
    grid.setWidget(2,1, msg);*/

    p.add(grid);

    passwordInput.addKeyboardListener(
        new KeyboardListener()
        {

          public void onKeyDown(Widget widget, char c, int i)
          {
          }

          public void onKeyPress(Widget widget, char c, int i)
          {
          }

          public void onKeyUp(Widget widget, char c, int i)
          {
            if(c == KeyboardListener.KEY_ENTER)
            {
              engageLogin();
            }
          }
        }
    );


    /*HTML htmlMsg = new HTML("Messages");
    p.add(htmlMsg);*/

    return p;
  }

}
