/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.engine;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.ui.ChangeListener;
import com.google.gwt.user.client.ui.ClickListener;
import com.google.gwt.user.client.ui.Widget;
import com.google.gwt.user.client.ui.PushButton;
import com.mvc4g.client.Controller;
import com.mvc4g.client.Event;
import org.gwt.mosaic.ui.client.ListBox;
import org.gwt.mosaic.ui.client.MessageBox;
import org.gwt.mosaic.ui.client.ToolBar;
import org.gwt.mosaic.ui.client.ToolButton;
import org.gwt.mosaic.ui.client.layout.*;
import org.gwt.mosaic.ui.client.list.DefaultListModel;
import org.jboss.bpm.console.client.common.AbstractView;
import org.jboss.bpm.console.client.icons.ConsoleIconBundle;
import org.jboss.bpm.console.client.model.DeploymentRef;
import org.jboss.bpm.console.client.util.SimpleDateFormat;

import java.util.Date;
import java.util.List;

/**
 * List of deployments
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class DeploymentListView extends AbstractView
{
  public final static String ID = DeploymentListView.class.getName();

  private Controller controller;

  private boolean initialized;

  private LayoutPanel deploymentList = null;

  private ListBox<DeploymentRef> listBox;

  private DeploymentRef selection = null;

  private SimpleDateFormat dateFormat = new SimpleDateFormat();

  private int FILTER_NONE       = 10;
  private int FILTER_ACTIVE     = 20;
  private int FILTER_SUSPENDED  = 30;
  private int currentFilter = FILTER_NONE;

  private List<DeploymentRef> deployments = null;

  private DeploymentDetailView detailView;

  public DeploymentListView()
  {
    super();
    ConsoleIconBundle icons = GWT.create(ConsoleIconBundle.class);
    setTitle("Deployments");
    setIcon(icons.deploymentIcon());

    listBox = createListBox();

  }

  private ListBox createListBox()
  {
    final ListBox<DeploymentRef> listBox =
        new ListBox<DeploymentRef>(
            new String[] {
                "ID", "Name", "Timestamp", "Suspended"}
        );


    listBox.setCellRenderer(new ListBox.CellRenderer<DeploymentRef>() {
      public void renderCell(ListBox<DeploymentRef> listBox, int row, int column,
                             DeploymentRef item) {
        switch (column) {
          case 0:
            listBox.setText(row, column, item.getId());
            break;
          case 1:
            listBox.setText(row, column, item.getName());
            break;
          case 2:
            listBox.setText(row, column, dateFormat.format(new Date(item.getTimestamp())));
            break;
          case 3:
            listBox.setText(row, column, String.valueOf(item.isSuspended()));
            break;
          default:
            throw new RuntimeException("Unexpected column size");
        }
      }
    });

    listBox.addChangeListener(new ChangeListener()
    {
      public void onChange(Widget widget)
      {
        int index = listBox.getSelectedIndex();
        if(index!=-1)
        {
          DeploymentRef item = listBox.getItem(index);

          controller.handleEvent(
              new Event(UpdateDeploymentDetailAction.ID, item)
          );
        }
      }
    });

    return listBox;
  }

  public void setController(Controller controller)
  {
    this.controller = controller;
  }

  public boolean isInitialized()
  {
    return initialized;
  }

  public void initialize()
  {
    if(!initialized)
    {
      deploymentList = new LayoutPanel( new BoxLayout(BoxLayout.Orientation.VERTICAL));
      deploymentList.setPadding(0);
      deploymentList.setWidgetSpacing(0);

      // toolbar

      final LayoutPanel toolBox = new LayoutPanel();
      toolBox.setPadding(0);
      toolBox.setWidgetSpacing(0);
      toolBox.setLayout(new BoxLayout(BoxLayout.Orientation.HORIZONTAL));

      final ToolBar toolBar = new ToolBar();
      toolBar.add(
          new PushButton("Refresh", new ClickListener() {
            public void onClick(Widget sender) {
              // force loading
              controller.handleEvent(
                  new Event(UpdateDeploymentsAction.ID, null)
              );
            }
          }
          )
      );


      toolBar.add(
          new PushButton("Delete", new ClickListener() {
            public void onClick(Widget sender) {

              DeploymentRef deploymentRef = getSelection();
              if(deploymentRef!=null)
              {
                MessageBox.confirm("Delete deployment",
                    "Do you want to delete this deployment? Any related data will be removed.",
                    new MessageBox.ConfirmationCallback() {
                      public void onResult(boolean doIt)
                      {
                        if(doIt)
                        {
                          controller.handleEvent(
                              new Event(
                                  DeleteDeploymentAction.ID,
                                  getSelection().getId()
                              )
                          );
                        }
                      }
                    });
              }
              else
              {
                MessageBox.alert("Missing selection", "Please select a deployment");
              }
            }
          }
          )
      );

      toolBox.add(toolBar, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));

      // filter
      LayoutPanel filterPanel = new LayoutPanel(new BoxLayout(BoxLayout.Orientation.VERTICAL));
      filterPanel.setStyleName("bpm-filter-panel");
      final com.google.gwt.user.client.ui.ListBox dropBox = new com.google.gwt.user.client.ui.ListBox(false);
      dropBox.setStyleName("bpm-operation-ui");
      dropBox.addItem("All");
      dropBox.addItem("Active");
      dropBox.addItem("Suspended");

      dropBox.addChangeListener(new ChangeListener() {
        public void onChange(Widget sender) {
          switch (dropBox.getSelectedIndex())
          {
            case 0:
              currentFilter = FILTER_NONE;
              break;
            case 1:
              currentFilter = FILTER_ACTIVE;
              break;
            case 2:
              currentFilter = FILTER_SUSPENDED;
              break;
            default:
              throw new IllegalArgumentException("No such index");
          }

          renderFiltered();
        }
      });
      filterPanel.add(dropBox);

      toolBox.add(filterPanel, new BoxLayoutData(BoxLayoutData.FillStyle.VERTICAL));

      this.deploymentList.add(toolBox, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));
      this.deploymentList.add(listBox, new BoxLayoutData(BoxLayoutData.FillStyle.BOTH));

      // layout
      LayoutPanel layout = new LayoutPanel(new BorderLayout());
      layout.add(deploymentList, new BorderLayoutData(BorderLayout.Region.CENTER));

      // details
      // detail panel
      detailView = new DeploymentDetailView();
      controller.addView(DeploymentDetailView.ID, detailView);
      layout.add(detailView, new BorderLayoutData(BorderLayout.Region.SOUTH,10,200));
  
      this.add(layout);

      initialized = true;
    }
  }

  public DeploymentRef getSelection()
  {
    DeploymentRef selection = null;
    if(isInitialized() && listBox.getSelectedIndex()!=-1)
      selection = listBox.getItem( listBox.getSelectedIndex());
    return selection;
  }

  public void update(List<DeploymentRef> deployments)
  {
    this.deployments = deployments;

    renderFiltered();
  }

  private void renderFiltered()
  {
    if(this.deployments!=null)
    {
      final DefaultListModel<DeploymentRef> model =
          (DefaultListModel<DeploymentRef>) listBox.getModel();

      model.clear();

      for(DeploymentRef dpl : deployments)
      {
        if(FILTER_NONE==currentFilter)
        {
          model.add(dpl);
        }
        else
        {
          boolean showSuspended = (FILTER_SUSPENDED==currentFilter);
          if(dpl.isSuspended()==showSuspended)
            model.add(dpl);
        }
      }

      if(listBox.getSelectedIndex()!=-1)
        listBox.setItemSelected(listBox.getSelectedIndex(), false);

      // clear details
      controller.handleEvent(
          new Event(UpdateDeploymentDetailAction.ID, null)
      );
    }
  }

  public void select(String deploymentId)
  {
    final DefaultListModel<DeploymentRef> model =
        (DefaultListModel<DeploymentRef>) listBox.getModel();

    for(int i=0; i<model.size(); i++)
    {
      DeploymentRef ref = model.get(i);
      if(ref.getId().equals(deploymentId))
      {
        listBox.setSelectedIndex(i);
        break;
      }
    }

  }
}
