/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.process;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.ui.ChangeListener;
import com.google.gwt.user.client.ui.ClickListener;
import com.google.gwt.user.client.ui.Widget;
import com.google.gwt.user.client.ui.PushButton;
import com.mvc4g.client.Controller;
import com.mvc4g.client.Event;
import org.gwt.mosaic.ui.client.ListBox;
import org.gwt.mosaic.ui.client.ToolBar;
import org.gwt.mosaic.ui.client.ToolButton;
import org.gwt.mosaic.ui.client.Caption;
import org.gwt.mosaic.ui.client.layout.*;
import org.gwt.mosaic.ui.client.list.DefaultListModel;
import org.jboss.bpm.console.client.common.AbstractView;
import org.jboss.bpm.console.client.common.PagingPanel;
import org.jboss.bpm.console.client.common.PagingCallback;
import org.jboss.bpm.console.client.icons.ConsoleIconBundle;
import org.jboss.bpm.console.client.model.ProcessDefinitionRef;

import java.util.ArrayList;
import java.util.List;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
class DefinitionListView extends AbstractView
{
  public final static String ID = DefinitionListView.class.getName();

  private Controller controller;

  private LayoutPanel definitionList = null;

  private ListBox<ProcessDefinitionRef> listBox;

  private boolean isInitialized;

  private int FILTER_NONE       = 10;
  private int FILTER_ACTIVE     = 20;
  private int FILTER_SUSPENDED  = 30;
  private int currentFilter = FILTER_NONE;

  private List<ProcessDefinitionRef> definitions = null;
  private PagingPanel pagingPanel;
  
  public DefinitionListView()
  {
    super();
    ConsoleIconBundle icons = GWT.create(ConsoleIconBundle.class);
    setTitle("Process Definitions");
    setIcon(icons.processIcon());

    listBox = createListBox();

  }

  public boolean isInitialized()
  {
    return isInitialized;
  }

  public void initialize()
  {
    if(!isInitialized)
    {
      
      definitionList = new LayoutPanel( new BoxLayout(BoxLayout.Orientation.VERTICAL));
      definitionList.setPadding(0);
      definitionList.setWidgetSpacing(0);

      // toolbar

      final LayoutPanel toolBox = new LayoutPanel();
      toolBox.setPadding(0);
      toolBox.setWidgetSpacing(0);
      toolBox.setLayout(new BoxLayout(BoxLayout.Orientation.HORIZONTAL));

      // toolbar
      final ToolBar toolBar = new ToolBar();
      toolBar.add(
          new PushButton("Refresh", new ClickListener() {
            public void onClick(Widget sender) {
              // force loading
              controller.handleEvent(
                  new Event(UpdateDefinitionsAction.ID, null)
              );
            }
          }
          )
      );
      toolBox.add(toolBar, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));

      // filter
      LayoutPanel filterPanel = new LayoutPanel(new BoxLayout(BoxLayout.Orientation.VERTICAL));
      filterPanel.setStyleName("bpm-filter-panel");
      final com.google.gwt.user.client.ui.ListBox dropBox = new com.google.gwt.user.client.ui.ListBox(false);
      dropBox.setStyleName("bpm-operation-ui");
      dropBox.addItem("All");
      dropBox.addItem("Active");
      dropBox.addItem("Suspended");

      dropBox.addChangeListener(new ChangeListener() {
        public void onChange(Widget sender) {
          switch (dropBox.getSelectedIndex())
          {
            case 0:
              currentFilter = FILTER_NONE;
              break;
            case 1:
              currentFilter = FILTER_ACTIVE;
              break;
            case 2:
              currentFilter = FILTER_SUSPENDED;
              break;
            default:
              throw new IllegalArgumentException("No such index");
          }

          renderFiltered();
        }
      });
      filterPanel.add(dropBox);

      toolBox.add(filterPanel, new BoxLayoutData(BoxLayoutData.FillStyle.VERTICAL));

      definitionList.add(toolBox, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));
      definitionList.add(listBox, new BoxLayoutData(BoxLayoutData.FillStyle.BOTH));
      pagingPanel = new PagingPanel(
          new PagingCallback()
          {
            public void rev()
            {
              renderFiltered();
            }

            public void ffw()
            {
              renderFiltered();
            }
          }
      );
      definitionList.add(pagingPanel,new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));

      // layout
      LayoutPanel layout = new LayoutPanel(new BorderLayout());
      layout.add(definitionList, new BorderLayoutData(BorderLayout.Region.CENTER));

      // details
      ProcessDetailView detailsView = new ProcessDetailView();
      controller.addView(ProcessDetailView.ID, detailsView);
      controller.addAction(UpdateProcessDetailAction.ID, new UpdateProcessDetailAction());
      layout.add(detailsView, new BorderLayoutData(BorderLayout.Region.SOUTH, 10,200));

      this.add(layout);

      isInitialized = true;
    }
  }

  private ListBox createListBox()
  {
    final ListBox<ProcessDefinitionRef> listBox =
        new ListBox<ProcessDefinitionRef>(
            new String[] {
                "Name", "Version", "Suspended"
            }
        );    
    
    listBox.setCellRenderer(new ListBox.CellRenderer<ProcessDefinitionRef>() {
      public void renderCell(ListBox<ProcessDefinitionRef> listBox, int row, int column,
                             ProcessDefinitionRef item) {
        switch (column) {
          case 0:
            listBox.setText(row, column, item.getName());            
            break;
          case 1:
            listBox.setText(row, column, String.valueOf(item.getVersion()));
            break;
          case 2:
            listBox.setText(row, column, String.valueOf(item.isSuspended()));
            break;
          default:
            throw new RuntimeException("Unexpected column size");
        }
      }
    });

    listBox.addChangeListener(new ChangeListener()
    {
      public void onChange(Widget widget)
      {
        int index = listBox.getSelectedIndex();
        if(index!=-1)
        {
          ProcessDefinitionRef item = listBox.getItem(index);

          // update details
          controller.handleEvent(
              new Event(UpdateProcessDetailAction.ID, item)
          );

          // load instances
          controller.handleEvent(
              new Event(
                  UpdateInstancesAction.ID,
                  item
              )
          );
        }
      }
    });
  
    return listBox;
  }


  public void setController(Controller controller)
  {
    this.controller = controller;
  }

  public void update(List<ProcessDefinitionRef> definitions)
  {
    this.definitions = definitions;
    pagingPanel.reset();
    renderFiltered();
  }

  private void renderFiltered()
  {
    if(this.definitions!=null)
    {
      final DefaultListModel<ProcessDefinitionRef> model =
          (DefaultListModel<ProcessDefinitionRef>) listBox.getModel();

      model.clear();

      List<ProcessDefinitionRef> tmp = new ArrayList<ProcessDefinitionRef>();
      for(ProcessDefinitionRef def : definitions)
      {
        if(FILTER_NONE==currentFilter)
        {
          tmp.add(def);
        }
        else
        {
          boolean showSuspended = (FILTER_SUSPENDED==currentFilter);
          if(def.isSuspended()==showSuspended)
            tmp.add(def);
        }
      }

      for(ProcessDefinitionRef def : (List<ProcessDefinitionRef>)pagingPanel.trim(tmp) )
        model.add(def);
      
      if(listBox.getSelectedIndex()!=-1)
        listBox.setItemSelected(listBox.getSelectedIndex(), false);

      // clear details
      controller.handleEvent(
          new Event(UpdateProcessDetailAction.ID, null)
      );
    }
  }

  public ProcessDefinitionRef getSelection()
  {
    ProcessDefinitionRef selection = null;
    if(isInitialized() && listBox.getSelectedIndex()!=-1)
      selection = listBox.getItem( listBox.getSelectedIndex());
    return selection;
  }

}
