/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.task;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.ui.ChangeListener;
import com.google.gwt.user.client.ui.ClickListener;
import com.google.gwt.user.client.ui.PushButton;
import com.google.gwt.user.client.ui.Widget;
import com.mvc4g.client.Event;
import org.gwt.mosaic.ui.client.ListBox;
import org.gwt.mosaic.ui.client.MessageBox;
import org.gwt.mosaic.ui.client.ToolBar;
import org.gwt.mosaic.ui.client.layout.*;
import org.gwt.mosaic.ui.client.list.DefaultListModel;
import org.jboss.bpm.console.client.ApplicationContext;
import org.jboss.bpm.console.client.ServerPlugins;
import org.jboss.bpm.console.client.common.IFrameWindowCallback;
import org.jboss.bpm.console.client.common.IFrameWindowPanel;
import org.jboss.bpm.console.client.common.PagingPanel;
import org.jboss.bpm.console.client.common.PagingCallback;
import org.jboss.bpm.console.client.icons.ConsoleIconBundle;
import org.jboss.bpm.console.client.model.TaskRef;
import org.jboss.bpm.console.client.task.events.DetailViewEvent;
import org.jboss.bpm.console.client.task.events.TaskIdentityEvent;
import org.jboss.bpm.console.client.util.SimpleDateFormat;

import java.util.List;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class AssignedTasksView extends AbstractTaskList
{

  public final static String ID = AssignedTasksView.class.getName();

  private final ApplicationContext appContext;

  private IFrameWindowPanel iframeWindow = null;
  
  private TaskDetailView detailsView;

  private SimpleDateFormat dateFormat = new SimpleDateFormat();

  private boolean hasDispatcherPlugin;

  private PagingPanel pagingPanel;

  public AssignedTasksView(ApplicationContext appContext, TaskDetailView detailsView)
  {
    super();
    this.appContext = appContext;
    ConsoleIconBundle icons = GWT.create(ConsoleIconBundle.class);
    setTitle("Personal Tasks");
    setIcon(icons.userIcon());

    this.detailsView = detailsView;
  }

  public void initialize()
  {
    if(!isInitialized)
    {
      taskList = new LayoutPanel( new BoxLayout(BoxLayout.Orientation.VERTICAL));
      taskList.setPadding(0);
      taskList.setWidgetSpacing(0);

      listBox =
          new ListBox<TaskRef>(
              new String[] {
                  "Priority", "Process", "Task Name", "Due Date"
              }
          );


      listBox.setCellRenderer(new ListBox.CellRenderer<TaskRef>() {
        public void renderCell(ListBox<TaskRef> listBox, int row, int column,
                               TaskRef item) {
          switch (column) {
            case 0:
              listBox.setText(row, column, String.valueOf(item.getPriority()));
              break;
            case 1:
              listBox.setText(row, column, item.getProcessId());
              break;
            case 2:
              listBox.setText(row, column, item.getName());
              break;
            case 3:
              String d = item.getDueDate() != null ? dateFormat.format(item.getDueDate()):"";
              listBox.setText(row, column, d);
              break;
            default:
              throw new RuntimeException("Unexpected column size");
          }
        }
      });

      listBox.addChangeListener(
          new ChangeListener() {

            public void onChange(Widget widget)
            {
              TaskRef task = getSelection(); // first call always null?
              if(task!=null)
              {
                controller.handleEvent(
                    new Event(UpdateDetailsAction.ID, new DetailViewEvent("AssignedDetailView", task))
                );
              }              
            }
          }
      );

      // toolbar
      final LayoutPanel toolBox = new LayoutPanel();
      toolBox.setPadding(0);
      toolBox.setWidgetSpacing(5);
      //toolBox.setLayout(new BoxLayout(BoxLayout.Orientation.VERTICAL));

      final ToolBar toolBar = new ToolBar();
      toolBar.add(
          new PushButton("Refresh", new ClickListener() {
            public void onClick(Widget sender) {
              // force loading
              controller.handleEvent(
                  new Event(LoadTasksAction.ID, appContext.getAuthentication().getUsername())
              );
            }
          }
          )
      );


      PushButton viewBtn = new PushButton("View", new ClickListener()
      {
        public void onClick(Widget sender)
        {

          TaskRef selection = getSelection();

          if (selection != null)
          {
            if (selection.getUrl() != null && !selection.getUrl().equals(""))
            {
              iframeWindow = new IFrameWindowPanel(
                  selection.getUrl(), "Task Form: "+selection.getName()
              );

              iframeWindow.setCallback(
                  new IFrameWindowCallback()
                  {
                    public void onWindowClosed()
                    {
                      controller.handleEvent(
                          new Event(LoadTasksAction.ID, appContext.getAuthentication().getUsername())
                      );
                    }
                  }
              );

              iframeWindow.show();
            }
            else
            {
              MessageBox.alert("Invalid operation", "The task doesn't provide a UI");
            }
          }
          else
          {
            MessageBox.alert("Missing selection", "Please select a task");
          }
        }
      }
      );
      toolBar.add(viewBtn);

      toolBar.add(
          new PushButton("Release", new ClickListener() {
            public void onClick(Widget sender) {

              TaskRef selection = getSelection();

              if(selection!=null)
              {
                TaskIdentityEvent payload = new TaskIdentityEvent(
                    null, selection
                );

                controller.handleEvent(
                    new Event(ReleaseTaskAction.ID, payload)
                );
              }
              else
              {
                MessageBox.alert("Missing selection", "Please select a task");
              }
            }
          }
          )
      );


      toolBox.add(toolBar, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));

      this.taskList.add(toolBox, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));
      this.taskList.add(listBox, new BoxLayoutData(BoxLayoutData.FillStyle.BOTH));

      pagingPanel = new PagingPanel(
          new PagingCallback()
          {
            public void rev()
            {
              renderUpdate();
            }

            public void ffw()
            {
              renderUpdate();
            }
          }
      );

      this.taskList.add(pagingPanel, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));
      
      LayoutPanel layout = new LayoutPanel(new BorderLayout());
      layout.add(taskList, new BorderLayoutData(BorderLayout.Region.CENTER));
      layout.add(detailsView, new BorderLayoutData(BorderLayout.Region.SOUTH, 10 , 200));

      this.add(layout);

      // plugin availability
      this.hasDispatcherPlugin =
          ServerPlugins.has("org.jboss.bpm.console.server.plugin.FormDispatcherPlugin");
      viewBtn.setEnabled(hasDispatcherPlugin);
      
      isInitialized = true;
    }
  }
  
  public void update(String identity, List<TaskRef> tasks)
  {
    this.identity = identity;
    this.cachedTasks = tasks;
    if(isInitialized())pagingPanel.reset();
    renderUpdate();
  }

  private void renderUpdate()
  {
    // lazy init
    initialize();

    final DefaultListModel<TaskRef> model =
        (DefaultListModel<TaskRef>) listBox.getModel();

    model.clear();

    List<TaskRef> trimmed = pagingPanel.trim(cachedTasks);
    for(TaskRef task : trimmed)
    {
      if(TaskRef.STATE.ASSIGNED ==task.getCurrentState())
        model.add(task);
    }

    // clear details
    controller.handleEvent(
        new Event(UpdateDetailsAction.ID, new DetailViewEvent("AssignedDetailView", null))
    );

    this.invalidate();

  }

}
