/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.task;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.ui.ChangeListener;
import com.google.gwt.user.client.ui.ClickListener;
import com.google.gwt.user.client.ui.PushButton;
import com.google.gwt.user.client.ui.Widget;
import com.mvc4g.client.Event;
import org.gwt.mosaic.ui.client.ListBox;
import org.gwt.mosaic.ui.client.MessageBox;
import org.gwt.mosaic.ui.client.ToolBar;
import org.gwt.mosaic.ui.client.layout.*;
import org.gwt.mosaic.ui.client.list.DefaultListModel;
import org.jboss.bpm.console.client.ApplicationContext;
import org.jboss.bpm.console.client.common.PagingCallback;
import org.jboss.bpm.console.client.common.PagingPanel;
import org.jboss.bpm.console.client.icons.ConsoleIconBundle;
import org.jboss.bpm.console.client.model.TaskRef;
import org.jboss.bpm.console.client.task.events.DetailViewEvent;
import org.jboss.bpm.console.client.task.events.TaskIdentityEvent;
import org.jboss.bpm.console.client.util.SimpleDateFormat;

import java.util.List;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
class OpenTasksView extends AbstractTaskList
{

  public final static String ID = OpenTasksView.class.getName();

  private TaskDetailView detailsView;

  private ApplicationContext appContext;

  private SimpleDateFormat dateFormat = new SimpleDateFormat();

  private PagingPanel pagingPanel;
  
  public OpenTasksView(ApplicationContext appContext, TaskDetailView detailView)
  {
    super();
    this.appContext = appContext;
    
    ConsoleIconBundle icons = GWT.create(ConsoleIconBundle.class);
    setTitle("Group Tasks");
    setIcon(icons.taskIcon());

    this.detailsView = detailView;
  }

  public void initialize()
  {
    if(!isInitialized)
    {
      taskList = new LayoutPanel( new BoxLayout(BoxLayout.Orientation.VERTICAL));
      taskList.setPadding(0);
      taskList.setWidgetSpacing(0);

      listBox =
          new ListBox<TaskRef>(
              new String[] {
                  "Priority", "Process", "Task Name", "Status", "Due Date"}
          );


      listBox.setCellRenderer(new ListBox.CellRenderer<TaskRef>() {
        public void renderCell(ListBox<TaskRef> listBox, int row, int column,
                               TaskRef item) {
          switch (column) {
            case 0:
              listBox.setText(row, column, String.valueOf(item.getPriority()));
              break;
            case 1:
              listBox.setText(row, column, item.getProcessId());
              break;
            case 2:
              listBox.setText(row, column, item.getName());
              break;
            case 3:
              listBox.setText(row, column, String.valueOf(item.getCurrentState()));
              break;
            case 4:
              String d = item.getDueDate() != null ? dateFormat.format(item.getDueDate()):"";
              listBox.setText(row, column, d);
              break;
            default:
              throw new RuntimeException("Unexpected column size");
          }
        }
      });


      listBox.addChangeListener(
          new ChangeListener() {

            public void onChange(Widget widget)
            {
              TaskRef task = getSelection(); // first call always null?
              if(task!=null)
              {
                controller.handleEvent(
                    new Event(UpdateDetailsAction.ID, new DetailViewEvent("OpenDetailView", task))
                );
              }             
            }
          }
      );

      // toolbar
      final LayoutPanel toolBox = new LayoutPanel();
      toolBox.setPadding(0);
      toolBox.setWidgetSpacing(5);

      final ToolBar toolBar = new ToolBar();
      toolBar.add(
          new PushButton("Refresh", new ClickListener() {
            public void onClick(Widget sender) {
              // force loading
              controller.handleEvent(
                  new Event(LoadTasksParticipationAction.ID, getAssignedIdentity())
              );
            }
          }
          )
      );   

      toolBar.add(
          new PushButton("Claim", new ClickListener() {
            public void onClick(Widget sender)
            {
              TaskRef selection = getSelection();

              if(selection!=null)
              {
                controller.handleEvent(
                    new Event(
                        ClaimTaskAction.ID,
                        new TaskIdentityEvent(appContext.getAuthentication().getUsername(), selection)
                    )
                );
              }
              else
              {
                MessageBox.alert("Missing selection", "Please select a task");
              }
            }
          }
          )
      );

      toolBox.add(toolBar, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));

      this.taskList.add(toolBox, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));
      this.taskList.add(listBox, new BoxLayoutData(BoxLayoutData.FillStyle.BOTH));

      pagingPanel = new PagingPanel(
          new PagingCallback()
          {
            public void rev()
            {
              renderUpdate();
            }

            public void ffw()
            {
              renderUpdate();
            }
          }
      );

      this.taskList.add(pagingPanel, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));
      
      // main layout
      LayoutPanel layout = new LayoutPanel(new BorderLayout());
      layout.add(taskList, new BorderLayoutData(BorderLayout.Region.CENTER));
      layout.add(detailsView, new BorderLayoutData(BorderLayout.Region.SOUTH, 10,200));

      this.add(layout);
      isInitialized = true;
    }
  }

  public void update(String identity, List<TaskRef> tasks)
  {
    this.identity = identity;
    this.cachedTasks = tasks;
    pagingPanel.reset();
    renderUpdate();
  }

  private void renderUpdate()
  {    
    // lazy init
    initialize();

    final DefaultListModel<TaskRef> model =
        (DefaultListModel<TaskRef>) listBox.getModel();

    model.clear();

    List<TaskRef> trimmed = pagingPanel.trim(cachedTasks);
    for(TaskRef task : trimmed)
    {
      if(TaskRef.STATE.OPEN ==task.getCurrentState())
        model.add(task);
    }

    // details
    controller.handleEvent(
        new Event(UpdateDetailsAction.ID, new DetailViewEvent("OpenDetailView", null))
    );
  }

}
