/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.engine;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.ui.ChangeListener;
import com.google.gwt.user.client.ui.ClickListener;
import com.google.gwt.user.client.ui.Widget;
import com.google.gwt.user.client.ui.PushButton;
import com.mvc4g.client.Controller;
import com.mvc4g.client.Event;
import org.gwt.mosaic.ui.client.ListBox;
import org.gwt.mosaic.ui.client.MessageBox;
import org.gwt.mosaic.ui.client.ToolBar;
import org.gwt.mosaic.ui.client.ToolButton;
import org.gwt.mosaic.ui.client.layout.*;
import org.gwt.mosaic.ui.client.list.DefaultListModel;
import org.jboss.bpm.console.client.ApplicationContext;
import org.jboss.bpm.console.client.common.AbstractView;
import org.jboss.bpm.console.client.icons.ConsoleIconBundle;
import org.jboss.bpm.console.client.model.JobRef;
import org.jboss.bpm.console.client.util.SimpleDateFormat;

import java.util.Date;
import java.util.List;

/**
 * Display a list of jobs waiting for execution.<br/>
 * I.e. pending Timers and Messages.
 *
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class JobListView extends AbstractView
{
  public final static String ID = JobListView.class.getName();

  private Controller controller;

  private LayoutPanel jobList = null;

  private ListBox<JobRef> listBox;

  private JobRef selection = null;

  private SimpleDateFormat dateFormat = new SimpleDateFormat();

  private int FILTER_NONE       = 10;
  private int FILTER_TIMER     = 20;
  private int FILTER_MESSAGE  = 30;
  private int currentFilter = FILTER_NONE;

  private List<JobRef> jobs = null;

  private ApplicationContext appContext;

  public JobListView(ApplicationContext appContext)
  {
    setTitle("Jobs");
    this.appContext = appContext;

    ConsoleIconBundle icons = GWT.create(ConsoleIconBundle.class);
    setIcon(icons.jobsIcon());

    this.listBox = createListBox();
  }

  private ListBox createListBox()
  {
    final ListBox<JobRef> listBox =
        new ListBox<JobRef>(
            new String[] {
                "ID", "Due Date", "Type"}
        );


    listBox.setCellRenderer(new ListBox.CellRenderer<JobRef>() {
      public void renderCell(ListBox<JobRef> listBox, int row, int column,
                             JobRef item) {
        switch (column) {
          case 0:
            listBox.setText(row, column, item.getId());
            break;
          case 1:
            long ts = item.getTimestamp();
            String ds = ts > 0 ?  dateFormat.format(new Date(ts)) : "";
            listBox.setText(row, column, ds);
            break;
          case 2:
            listBox.setText(row, column, item.getType());
            break;
          default:
            throw new RuntimeException("Unexpected column size");
        }
      }
    });

    listBox.addChangeListener(new ChangeListener()
    {
      public void onChange(Widget widget)
      {
        int index = listBox.getSelectedIndex();
        if(index!=-1)
        {
          JobRef item = listBox.getItem(index);

          /*controller.handleEvent(
              new Event(UpdateJobDetailAction.ID, item)
          );*/
        }
      }
    });

    return listBox;
  }

  public void setController(Controller controller)
  {
    this.controller = controller;
  }

  public boolean isInitialized()
  {
    return initialized;
  }

  public void initialize()
  {
    if(!initialized)
    {
      jobList = new LayoutPanel( new BoxLayout(BoxLayout.Orientation.VERTICAL));
      jobList.setPadding(0);
      jobList.setWidgetSpacing(0);

      // toolbar

      final LayoutPanel toolBox = new LayoutPanel();
      toolBox.setPadding(0);
      toolBox.setWidgetSpacing(0);
      toolBox.setLayout(new BoxLayout(BoxLayout.Orientation.HORIZONTAL));

      // toolbar
      final ToolBar toolBar = new ToolBar();
      toolBar.add(
          new PushButton("Refresh", new ClickListener() {
            public void onClick(Widget sender) {
              // force loading
              controller.handleEvent(
                  new Event(UpdateJobsAction.ID, null)
              );
            }
          }
          )
      );

      toolBar.add(
          new PushButton("Execute", new ClickListener() {
            public void onClick(Widget sender) {
              JobRef selection = getSelection();
              if(null==selection)
              {
                MessageBox.alert("Missing selection", "Please select a job!");
              }
              else
              {
                controller.handleEvent(
                    new Event(ExecuteJobAction.ID, selection.getId())
                );
              }
            }
          }
          )
      );

      toolBox.add(toolBar, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));

      // filter
      LayoutPanel filterPanel = new LayoutPanel(new BoxLayout(BoxLayout.Orientation.VERTICAL));
      filterPanel.setStyleName("bpm-filter-panel");
      final com.google.gwt.user.client.ui.ListBox dropBox = new com.google.gwt.user.client.ui.ListBox(false);
      dropBox.setStyleName("bpm-operation-ui");
      dropBox.addItem("All");
      dropBox.addItem("Timers");
      dropBox.addItem("Messages");

      dropBox.addChangeListener(new ChangeListener() {
        public void onChange(Widget sender) {
          switch (dropBox.getSelectedIndex())
          {
            case 0:
              currentFilter = FILTER_NONE;
              break;
            case 1:
              currentFilter = FILTER_TIMER;
              break;
            case 2:
              currentFilter = FILTER_MESSAGE;
              break;
            default:
              throw new IllegalArgumentException("No such index");
          }

          renderFiltered();
        }
      });
      filterPanel.add(dropBox);

      toolBox.add(filterPanel, new BoxLayoutData(BoxLayoutData.FillStyle.VERTICAL));

      this.jobList.add(toolBox, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));
      this.jobList.add(listBox, new BoxLayoutData(BoxLayoutData.FillStyle.BOTH));

      // layout
      LayoutPanel layout = new LayoutPanel(new BorderLayout());
      layout.add(jobList, new BorderLayoutData(BorderLayout.Region.CENTER));

      // details
      /*JobDetailView detailsView = new JobDetailView();
      controller.addView(JobDetailView.ID, detailsView);
      controller.addAction(UpdateJobDetailAction.ID, new UpdateJobDetailAction());
      layout.add(detailsView, new BorderLayoutData(BorderLayout.Region.SOUTH, 10,200));
      */

      controller.addAction(UpdateJobsAction.ID, new UpdateJobsAction());
      
      this.add(layout);

      this.initialized = true;
    }
  }

  public void update(List<JobRef> jobs)
  {
    this.jobs = jobs;

    renderFiltered();
  }

  private void renderFiltered()
  {
    if(this.jobs!=null)
    {
      final DefaultListModel<JobRef> model =
          (DefaultListModel<JobRef>) listBox.getModel();

      model.clear();

      for(JobRef def : jobs)
      {
        if(FILTER_NONE==currentFilter)
        {
          model.add(def);
        }
        else
        {
          if(FILTER_TIMER==currentFilter
              && def.getType().equals("timer"))
          {
            model.add(def);
          }
          else if(FILTER_MESSAGE==currentFilter
              && def.getType().equals("message"))
          {
            model.add(def);                  
          }
        }
      }

      if(listBox.getSelectedIndex()!=-1)
        listBox.setItemSelected(listBox.getSelectedIndex(), false);

      // clear details
     /* controller.handleEvent(
          new Event(UpdateJobDetailAction.ID, null)
      ); */
    }
  }

  public JobRef getSelection()
  {
    JobRef selection = null;
    if(isInitialized() && listBox.getSelectedIndex()!=-1)
      selection = listBox.getItem( listBox.getSelectedIndex());
    return selection;
  }
}
