/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.client.process;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.ui.*;
import com.mvc4g.client.Controller;
import com.mvc4g.client.Event;
import org.gwt.mosaic.ui.client.ListBox;
import org.gwt.mosaic.ui.client.MessageBox;
import org.gwt.mosaic.ui.client.ToolBar;
import org.gwt.mosaic.ui.client.ToolButton;
import org.gwt.mosaic.ui.client.layout.*;
import org.gwt.mosaic.ui.client.list.DefaultListModel;
import org.jboss.bpm.console.client.ApplicationContext;
import org.jboss.bpm.console.client.common.*;
import org.jboss.bpm.console.client.icons.ConsoleIconBundle;
import org.jboss.bpm.console.client.model.ProcessDefinitionRef;
import org.jboss.bpm.console.client.model.ProcessInstanceRef;
import org.jboss.bpm.console.client.process.events.InstanceEvent;
import org.jboss.bpm.console.client.util.SimpleDateFormat;

import java.util.List;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
class InstanceListView extends AbstractView
{
  public final static String ID = InstanceListView.class.getName();

  private Controller controller;

  private LayoutPanel instanceList = null;

  private ListBox<ProcessInstanceRef> listBox;

  private ProcessDefinitionRef currentDefinition;

  private boolean isInitialized;

  private List<ProcessInstanceRef> cachedInstances = null;

  private SimpleDateFormat dateFormat = new SimpleDateFormat();

  private ApplicationContext appContext;

  private IFrameWindowPanel iframeWindow = null;

  private boolean isRiftsawInstance;

  private PagingPanel pagingPanel;
  
  public InstanceListView(ApplicationContext appContext)
  {
    super();
    this.appContext = appContext;

    ConsoleIconBundle icons = GWT.create(ConsoleIconBundle.class);
    setTitle("Process Instances");
    setIcon(icons.instanceIcon());

    // riftsaw?
    isRiftsawInstance = appContext.getConfig().getProfileName().equals("BPEL Console");
  }

  public boolean isInitialized()
  {
    return isInitialized;
  }

  public void initialize()
  {
    if(!isInitialized)
    {
      instanceList = new LayoutPanel( new BoxLayout(BoxLayout.Orientation.VERTICAL));
      instanceList.setPadding(0);
      instanceList.setWidgetSpacing(0);
      
      listBox =
          new ListBox<ProcessInstanceRef>(
              new String[] {
                  "Instance ID", "State", "Start Date"}
          );

      listBox.setCellRenderer(new ListBox.CellRenderer<ProcessInstanceRef>() {
        public void renderCell(ListBox<ProcessInstanceRef> listBox, int row, int column,
                               ProcessInstanceRef item) {
          switch (column) {
            case 0:
              listBox.setText(row, column, item.getId());
              break;
            case 1:
              listBox.setText(row, column, item.getState().toString());
              break;
            case 2:
              String d = item.getStartDate() != null ? dateFormat.format(item.getStartDate()) : "";
              listBox.setText(row, column, d);
              break;
            default:
              throw new RuntimeException("Unexpected column size");
          }
        }
      });

      listBox.addChangeListener(new ChangeListener()
      {
        public void onChange(Widget widget)
        {
          int index = listBox.getSelectedIndex();
          if(index!=-1)
          {
            ProcessInstanceRef item = listBox.getItem(index);

            // update details
            controller.handleEvent(
                new Event(UpdateInstanceDetailAction.ID,
                    new InstanceEvent(currentDefinition, item)
                )
            );
          }
        }
      });

      // toolbar
      final LayoutPanel toolBox = new LayoutPanel();
      toolBox.setPadding(0);
      toolBox.setWidgetSpacing(5);

      final ToolBar toolBar = new ToolBar();
      toolBar.add(
          new PushButton("Refresh", new ClickListener() {
            public void onClick(Widget sender) {
              controller.handleEvent(
                  new Event(
                      UpdateInstancesAction.ID,
                      getCurrentDefinition()
                  )
              );
            }
          }
          )
      );



      toolBar.add(
          new PushButton("Start", new ClickListener()
          {
            public void onClick(Widget sender)
            {
              MessageBox.confirm("Start new execution",
                  "Do you want to start a new execution of this process?",
                  new MessageBox.ConfirmationCallback() {
                    public void onResult(boolean doIt)
                    {
                      if(doIt)
                      {
                        String url = getCurrentDefinition().getFormUrl();
                        boolean hasForm = (url !=null && !url.equals(""));
                        if(hasForm)
                        {
                          ProcessDefinitionRef definition = getCurrentDefinition();
                          iframeWindow = new IFrameWindowPanel(
                              definition.getFormUrl(), "New Process Instance: "+ definition.getId()
                          );

                          iframeWindow.setCallback(
                              new IFrameWindowCallback()
                              {
                                public void onWindowClosed()
                                {
                                  controller.handleEvent(
                                      new Event(UpdateInstancesAction.ID, getCurrentDefinition())
                                  );
                                }
                              }
                          );

                          iframeWindow.show();
                        }
                        else
                        {
                          controller.handleEvent(
                              new Event(
                                  StartNewInstanceAction.ID,
                                  getCurrentDefinition()
                              )
                          );
                        }
                      }

                    }
                  });

            }
          }
          )
          {{
            setVisible(!isRiftsawInstance);
          }}
      );



      toolBar.add(
          new PushButton("Terminate", new ClickListener()
          {
            public void onClick(Widget sender)
            {

              if(getSelection()!=null)
              {

                MessageBox.confirm("Terminate instance",
                    "Terminating this instance will stop further execution.",
                    new MessageBox.ConfirmationCallback() {
                      public void onResult(boolean doIt)
                      {
                        if(doIt)
                        {
                          ProcessInstanceRef selection = getSelection();
                          selection.setState(ProcessInstanceRef.STATE.ENDED);
                          selection.setEndResult(ProcessInstanceRef.RESULT.OBSOLETE);
                          controller.handleEvent(
                              new Event(
                                  StateChangeAction.ID,
                                  selection
                              )
                          );
                        }
                      }
                    });
              }
              else
              {
                MessageBox.alert("Missing selection", "Please select an instance");
              }
            }
          }
          )
      );



      toolBar.add(
          new PushButton("Delete", new ClickListener()
          {
            public void onClick(Widget sender)
            {

              if(getSelection()!=null)
              {
                MessageBox.confirm("Delete instance",
                    "Deleting this instance will remove any history information and associated tasks as well.",
                    new MessageBox.ConfirmationCallback() {
                      public void onResult(boolean doIt)
                      {

                        if(doIt)
                        {
                          ProcessInstanceRef selection = getSelection();
                          selection.setState(ProcessInstanceRef.STATE.ENDED);

                          controller.handleEvent(
                              new Event(
                                  DeleteInstanceAction.ID,
                                  selection
                              )
                          );
                        }
                      }
                    });

              }
              else
              {
                MessageBox.alert("Missing selection", "Please select an instance");
              }
            }
          }
          )
          {{
            setVisible(!isRiftsawInstance);
          }}
      );

      toolBox.add(toolBar, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));

      instanceList.add(toolBox, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));
      instanceList.add(listBox, new BoxLayoutData(BoxLayoutData.FillStyle.BOTH));

      pagingPanel = new PagingPanel(
          new PagingCallback()
          {
            public void rev()
            {
              renderUpdate();
            }

            public void ffw()
            {
              renderUpdate();
            }
          }
      );
      instanceList.add(pagingPanel, new BoxLayoutData(BoxLayoutData.FillStyle.HORIZONTAL));

      // cached data?
      if(this.cachedInstances!=null)
        bindData(this.cachedInstances);

      // layout
      LayoutPanel layout = new LayoutPanel(new BorderLayout());
      layout.add(instanceList, new BorderLayoutData(BorderLayout.Region.CENTER));

      // details
      InstanceDetailView detailsView = new InstanceDetailView(appContext);
      controller.addView(InstanceDetailView.ID, detailsView);
      controller.addAction(UpdateInstanceDetailAction.ID, new UpdateInstanceDetailAction());
      layout.add(detailsView, new BorderLayoutData(BorderLayout.Region.SOUTH,10,200));

      this.add(layout);

      isInitialized = true;

    }
  }

  public ProcessInstanceRef getSelection()
  {
    ProcessInstanceRef selection = null;
    if(listBox.getSelectedIndex()!=-1)
      selection = listBox.getItem( listBox.getSelectedIndex());
    return selection;
  }

  public ProcessDefinitionRef getCurrentDefinition()
  {
    return this.currentDefinition;
  }

  public void setController(Controller controller)
  {
    this.controller = controller;
  }


  public void update(final ProcessDefinitionRef def, List<ProcessInstanceRef> instances)
  {
    this.currentDefinition = def;
    this.cachedInstances = instances;
    
    if(isInitialized()) pagingPanel.reset();
    renderUpdate();
  }

  private void renderUpdate()
  {
    if(isInitialized())
    {
      bindData(this.cachedInstances);

      // clear details
      controller.handleEvent(
          new Event(UpdateInstanceDetailAction.ID,
              new InstanceEvent(this.currentDefinition, null)
          )
      );

    }
  }

  private void bindData(List<ProcessInstanceRef> instances)
  {
    final DefaultListModel<ProcessInstanceRef> model =
        (DefaultListModel<ProcessInstanceRef>) listBox.getModel();
    model.clear();

    List<ProcessInstanceRef> list = pagingPanel.trim(instances);
    for(ProcessInstanceRef inst : list)
    {
      model.add(inst);
    }

    // layout again
    this.invalidate();
  }

}
