/*
 * Decompiled with CFR 0.152.
 */
package com.sleepycat.je.log;

import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.EnvironmentStats;
import com.sleepycat.je.RunRecoveryException;
import com.sleepycat.je.StatsConfig;
import com.sleepycat.je.cleaner.TrackedFileSummary;
import com.sleepycat.je.cleaner.UtilizationTracker;
import com.sleepycat.je.config.EnvironmentParams;
import com.sleepycat.je.dbi.DbConfigManager;
import com.sleepycat.je.dbi.EnvironmentImpl;
import com.sleepycat.je.latch.Latch;
import com.sleepycat.je.log.CheckpointMonitor;
import com.sleepycat.je.log.ChecksumValidator;
import com.sleepycat.je.log.FileHandleSource;
import com.sleepycat.je.log.FileManager;
import com.sleepycat.je.log.FileSource;
import com.sleepycat.je.log.LogBuffer;
import com.sleepycat.je.log.LogBufferPool;
import com.sleepycat.je.log.LogEntryType;
import com.sleepycat.je.log.LogException;
import com.sleepycat.je.log.LogFileNotFoundException;
import com.sleepycat.je.log.LogSource;
import com.sleepycat.je.log.LogUtils;
import com.sleepycat.je.log.LoggableObject;
import com.sleepycat.je.log.entry.LogEntry;
import com.sleepycat.je.utilint.Adler32;
import com.sleepycat.je.utilint.DbLsn;
import com.sleepycat.je.utilint.Tracer;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.BufferOverflowException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;

public abstract class LogManager {
    private static final String DEBUG_NAME;
    private static final boolean DEBUG = false;
    static final int HEADER_BYTES = 14;
    private static final int CHECKSUM_BYTES = 4;
    private static final int PREV_BYTES = 4;
    static final int HEADER_CONTENT_BYTES = 10;
    static final int HEADER_CHECKSUM_OFFSET = 0;
    static final int HEADER_ENTRY_TYPE_OFFSET = 4;
    static final int HEADER_VERSION_OFFSET = 5;
    static final int HEADER_PREV_OFFSET = 6;
    static final int HEADER_SIZE_OFFSET = 10;
    protected LogBufferPool logBufferPool;
    protected Latch logWriteLatch;
    private boolean doChecksumOnRead;
    private FileManager fileManager;
    private CheckpointMonitor checkpointMonitor;
    protected EnvironmentImpl envImpl;
    private boolean readOnly;
    private int readBufferSize;
    private int nRepeatFaultReads;
    static final /* synthetic */ boolean $assertionsDisabled;

    public LogManager(EnvironmentImpl envImpl, boolean readOnly) throws DatabaseException {
        this.envImpl = envImpl;
        this.fileManager = envImpl.getFileManager();
        DbConfigManager configManager = envImpl.getConfigManager();
        this.readOnly = readOnly;
        this.logBufferPool = new LogBufferPool(this.fileManager, envImpl);
        this.doChecksumOnRead = configManager.getBoolean(EnvironmentParams.LOG_CHECKSUM_READ);
        this.logWriteLatch = new Latch(DEBUG_NAME, envImpl);
        this.readBufferSize = configManager.getInt(EnvironmentParams.LOG_FAULT_READ_SIZE);
        this.checkpointMonitor = new CheckpointMonitor(envImpl);
    }

    public void resetPool(DbConfigManager configManager) throws DatabaseException {
        this.logBufferPool.reset(configManager);
    }

    public long logForceFlush(LoggableObject item, boolean fsyncRequired) throws DatabaseException {
        return this.log(item, false, true, fsyncRequired, -1L, false);
    }

    public long log(LoggableObject item) throws DatabaseException {
        return this.log(item, false, false, false, -1L, false);
    }

    public long log(LoggableObject item, boolean isProvisional, long oldNodeLsn, boolean isDeletedNode) throws DatabaseException {
        return this.log(item, isProvisional, false, false, oldNodeLsn, isDeletedNode);
    }

    private long log(LoggableObject item, boolean isProvisional, boolean flushRequired, boolean fsyncRequired, long oldNodeLsn, boolean isDeletedNode) throws DatabaseException {
        if (this.readOnly) {
            return -1L;
        }
        int itemSize = item.getLogSize();
        int entrySize = itemSize + 14;
        boolean marshallOutsideLatch = item.marshallOutsideWriteLatch();
        ByteBuffer marshalledBuffer = null;
        UtilizationTracker tracker = this.envImpl.getUtilizationTracker();
        LogResult logResult = null;
        try {
            if (marshallOutsideLatch) {
                marshalledBuffer = this.marshallIntoBuffer(item, itemSize, isProvisional, entrySize);
            }
            logResult = this.logItem(item, isProvisional, flushRequired, oldNodeLsn, isDeletedNode, entrySize, itemSize, marshallOutsideLatch, marshalledBuffer, tracker);
        }
        catch (BufferOverflowException e) {
            throw new RunRecoveryException(this.envImpl, (Throwable)e);
        }
        catch (IOException e) {
            throw new DatabaseException(Tracer.getStackTrace(e), e);
        }
        if (fsyncRequired) {
            this.fileManager.groupSync();
        }
        if (logResult.wakeupCheckpointer) {
            this.checkpointMonitor.activate();
        }
        if (logResult.wakeupCleaner) {
            tracker.activateCleaner();
        }
        return logResult.currentLsn;
    }

    protected abstract LogResult logItem(LoggableObject var1, boolean var2, boolean var3, long var4, boolean var6, int var7, int var8, boolean var9, ByteBuffer var10, UtilizationTracker var11) throws IOException, DatabaseException;

    protected LogResult logInternal(LoggableObject item, boolean isProvisional, boolean flushRequired, long oldNodeLsn, boolean isDeletedNode, int entrySize, int itemSize, boolean marshallOutsideLatch, ByteBuffer marshalledBuffer, UtilizationTracker tracker) throws IOException, DatabaseException {
        boolean flippedFile = this.fileManager.bumpLsn(entrySize);
        long currentLsn = this.fileManager.getLastUsedLsn();
        LogEntryType entryType = item.getLogType();
        boolean wakeupCleaner = tracker.countNewLogEntry(currentLsn, entryType, entrySize);
        if (isDeletedNode) {
            tracker.countObsoleteNode(currentLsn, entryType, true);
        }
        if (oldNodeLsn != -1L) {
            tracker.countObsoleteNode(oldNodeLsn, entryType, true);
        }
        if (!marshallOutsideLatch) {
            marshalledBuffer = this.marshallIntoBuffer(item, itemSize, isProvisional, entrySize);
        }
        ByteBuffer useBuffer = this.logBufferPool.getWriteBuffer(entrySize, flippedFile);
        marshalledBuffer = this.addPrevOffsetAndChecksum(marshalledBuffer, this.fileManager.getPrevEntryOffset(), entrySize);
        useBuffer.put(marshalledBuffer);
        this.logBufferPool.writeCompleted(currentLsn, flushRequired);
        item.postLogWork(currentLsn);
        boolean wakeupCheckpointer = this.checkpointMonitor.recordLogWrite(entrySize, item);
        return new LogResult(currentLsn, wakeupCheckpointer, wakeupCleaner);
    }

    private ByteBuffer marshallIntoBuffer(LoggableObject item, int itemSize, boolean isProvisional, int entrySize) throws DatabaseException {
        ByteBuffer destBuffer = ByteBuffer.allocate(entrySize);
        destBuffer.position(4);
        this.writeHeader(destBuffer, item.getLogType(), itemSize, isProvisional);
        item.writeToLog(destBuffer);
        return destBuffer;
    }

    private ByteBuffer addPrevOffsetAndChecksum(ByteBuffer destBuffer, long lastOffset, int entrySize) {
        Adler32 checksum = new Adler32();
        destBuffer.position(6);
        LogUtils.writeUnsignedInt(destBuffer, lastOffset);
        checksum.update(destBuffer.array(), 4, entrySize - 4);
        destBuffer.position(0);
        LogUtils.putUnsignedInt(destBuffer, checksum.getValue());
        destBuffer.position(0);
        return destBuffer;
    }

    ByteBuffer putIntoBuffer(LoggableObject item, int itemSize, long prevLogEntryOffset, boolean isProvisional, int entrySize) throws DatabaseException {
        ByteBuffer destBuffer = this.marshallIntoBuffer(item, itemSize, isProvisional, entrySize);
        destBuffer.position(0);
        return this.addPrevOffsetAndChecksum(destBuffer, 0L, entrySize);
    }

    private void writeHeader(ByteBuffer destBuffer, LogEntryType itemType, int itemSize, boolean isProvisional) {
        byte typeNum = itemType.getTypeNum();
        destBuffer.put(typeNum);
        byte version = itemType.getVersion();
        if (isProvisional) {
            version = LogEntryType.setProvisional(version);
        }
        destBuffer.put(version);
        destBuffer.position(10);
        LogUtils.writeInt(destBuffer, itemSize);
    }

    public LogEntry getLogEntry(long lsn) throws DatabaseException {
        LogSource logSource = this.getLogSource(lsn);
        return this.getLogEntryFromLogSource(lsn, logSource);
    }

    LogEntry getLogEntry(long lsn, RandomAccessFile file) throws DatabaseException {
        return this.getLogEntryFromLogSource(lsn, new FileSource(file, this.readBufferSize));
    }

    private LogEntry getLogEntryFromLogSource(long lsn, LogSource logSource) throws DatabaseException {
        try {
            long fileOffset = DbLsn.getFileOffset(lsn);
            ByteBuffer entryBuffer = logSource.getBytes(fileOffset);
            ChecksumValidator validator = null;
            long storedChecksum = LogUtils.getUnsignedInt(entryBuffer);
            if (this.doChecksumOnRead) {
                validator = new ChecksumValidator();
                validator.update(this.envImpl, entryBuffer, 10);
            }
            byte loggableType = entryBuffer.get();
            byte version = entryBuffer.get();
            entryBuffer.position(entryBuffer.position() + 4);
            int itemSize = LogUtils.readInt(entryBuffer);
            if (entryBuffer.remaining() < itemSize) {
                entryBuffer = logSource.getBytes(fileOffset + 14L, itemSize);
                ++this.nRepeatFaultReads;
            }
            if (this.doChecksumOnRead) {
                validator.update(this.envImpl, entryBuffer, itemSize);
                validator.validate(this.envImpl, storedChecksum, lsn);
            }
            if (!$assertionsDisabled && !LogEntryType.isValidType(loggableType)) {
                throw new AssertionError((Object)("Read non-valid log entry type: " + loggableType));
            }
            LogEntry logEntry = LogEntryType.findType(loggableType, version).getNewLogEntry();
            logEntry.readEntry(entryBuffer, itemSize, version, true);
            LogEntry logEntry2 = logEntry;
            return logEntry2;
        }
        catch (DatabaseException e) {
            throw e;
        }
        catch (ClosedChannelException e) {
            throw new RunRecoveryException(this.envImpl, "Channel closed, may be due to thread interrupt", e);
        }
        catch (Exception e) {
            throw new DatabaseException(e);
        }
        finally {
            if (logSource != null) {
                logSource.release();
            }
        }
    }

    public Object get(long lsn) throws DatabaseException {
        LogEntry entry = this.getLogEntry(lsn);
        return entry.getMainItem();
    }

    private LogSource getLogSource(long lsn) throws DatabaseException {
        LogBuffer logBuffer = this.logBufferPool.getReadBuffer(lsn);
        if (logBuffer == null) {
            try {
                return new FileHandleSource(this.fileManager.getFileHandle(DbLsn.getFileNumber(lsn)), this.readBufferSize);
            }
            catch (LogFileNotFoundException e) {
                throw new LogFileNotFoundException(DbLsn.getNoFormatString(lsn) + ' ' + e.getMessage());
            }
        }
        return logBuffer;
    }

    public void flush() throws DatabaseException {
        if (this.readOnly) {
            return;
        }
        this.flushInternal();
        this.fileManager.syncLogEnd();
    }

    protected abstract void flushInternal() throws LogException, DatabaseException;

    public void loadStats(StatsConfig config, EnvironmentStats stats) throws DatabaseException {
        stats.setNRepeatFaultReads(this.nRepeatFaultReads);
        if (config.getClear()) {
            this.nRepeatFaultReads = 0;
        }
        this.logBufferPool.loadStats(config, stats);
        this.fileManager.loadStats(config, stats);
    }

    public abstract void countObsoleteNodes(long var1, LogEntryType var3, boolean var4, long var5, LogEntryType var7, boolean var8) throws DatabaseException;

    protected void countObsoleteNodesInternal(UtilizationTracker tracker, long lsn1, LogEntryType type1, boolean obsolete1, long lsn2, LogEntryType type2, boolean obsolete2) throws DatabaseException {
        if (lsn1 != -1L) {
            tracker.countObsoleteNode(lsn1, type1, obsolete1);
        }
        if (lsn2 != -1L) {
            tracker.countObsoleteNode(lsn2, type2, obsolete2);
        }
    }

    public abstract void countObsoleteNodes(TrackedFileSummary[] var1) throws DatabaseException;

    protected void countObsoleteNodesInternal(UtilizationTracker tracker, TrackedFileSummary[] summaries) throws DatabaseException {
        for (int i = 0; i < summaries.length; ++i) {
            TrackedFileSummary summary = summaries[i];
            tracker.addSummary(summary.getFileNumber(), summary);
        }
    }

    static {
        $assertionsDisabled = !LogManager.class.desiredAssertionStatus();
        DEBUG_NAME = LogManager.class.getName();
    }

    static class LogResult {
        long currentLsn;
        boolean wakeupCheckpointer;
        boolean wakeupCleaner;

        LogResult(long currentLsn, boolean wakeupCheckpointer, boolean wakeupCleaner) {
            this.currentLsn = currentLsn;
            this.wakeupCheckpointer = wakeupCheckpointer;
            this.wakeupCleaner = wakeupCleaner;
        }
    }
}

