/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */

package org.jboss.cache.pojo.jmx;

import static org.testng.AssertJUnit.assertEquals;
import static org.testng.AssertJUnit.assertFalse;
import static org.testng.AssertJUnit.assertTrue;
import static org.testng.AssertJUnit.fail;

import javax.management.ObjectName;

import org.jboss.cache.CacheException;
import org.jboss.cache.CacheStatus;
import org.jboss.cache.config.Configuration;
import org.jboss.cache.notifications.annotation.CacheListener;
import org.jboss.cache.notifications.annotation.CacheStarted;
import org.jboss.cache.notifications.annotation.CacheStopped;
import org.jboss.cache.notifications.event.Event;
import org.jboss.cache.pojo.PojoCacheException;
import org.testng.annotations.Test;

/**
 * Tests the PojoCacheJmxWrapper class
 *
 * @author <a href="mailto:ben.wang@jboss.org">Ben Wang</a>
 * @author Brian Stansberry
 */
@Test(groups = {"functional"})
public class PojoCacheJmxWrapperTest extends PojoCacheJmxWrapperTestBase
{

   public void testCacheMBeanBinding1() throws Exception
   {
      registerWrapper();
      assertTrue("Cache Mbean should be registered ", mBeanServer.isRegistered(plainCacheMBeanName));
      assertTrue("PojoCache Mbean should be registered ", mBeanServer.isRegistered(mBeanName));

      unregisterWrapper();
      assertFalse("Cache Mbean should not be registered ", mBeanServer.isRegistered(plainCacheMBeanName));
      assertFalse("PojoCache Mbean should not be registered ", mBeanServer.isRegistered(mBeanName));
   }

   public void testCacheMBeanBinding2() throws Exception
   {
      PojoCacheJmxWrapperMBean wrapper = createWrapper(createConfiguration());
      wrapper = registerWrapper(wrapper);
      assertFalse("Cache Mbean should not be registered ", mBeanServer.isRegistered(plainCacheMBeanName));
      assertTrue("PojoCache Mbean should be registered ", mBeanServer.isRegistered(mBeanName));

      wrapper.create();
      wrapper.start();

      assertTrue("Cache Mbean should be registered ", mBeanServer.isRegistered(plainCacheMBeanName));

      unregisterWrapper();
      assertFalse("Cache Mbean should not be registered ", mBeanServer.isRegistered(plainCacheMBeanName));
      assertFalse("PojoCache Mbean should not be registered ", mBeanServer.isRegistered(mBeanName));
   }

   public void testConfiguration() throws Exception
   {
      registerWrapper();
      Configuration cfgFromJmx = (Configuration) mBeanServer.getAttribute(plainCacheMBeanName, "Configuration");
      assertEquals(cache.getCache().getConfiguration(), cfgFromJmx);
   }

   public void testGetUnderlyingCacheObjectName() throws Exception
   {
      registerWrapper();
      String cacheName = (String) mBeanServer.getAttribute(mBeanName, "UnderlyingCacheObjectName");
      ObjectName tmpName = JmxUtil.getPlainCacheObjectName(mBeanName);
      assertEquals("Cache object name ", tmpName.getCanonicalName(), cacheName);
   }

   /**
    * Tests that setting registerPlainCache=false disables interceptor
    * registration when the wrapper is registered before create/start
    * are called.
    *
    * @throws Exception
    */
   public void testRegisterPlainCache1() throws Exception
   {
      PojoCacheJmxWrapper wrapper = createWrapper(createConfiguration());
      wrapper.setRegisterPlainCache(false);

      registerWrapper(wrapper);

      assertTrue("Should be registered", mBeanServer.isRegistered(mBeanName));
      assertFalse("Plain cache should not be registered", mBeanServer.isRegistered(plainCacheMBeanName));
   }

   /**
    * Tests that setting registerPlainCache=false disables interceptor
    * registration when the wrapper is registered after create/start
    * are called.
    *
    * @throws Exception
    */
   public void testRegisterPlainCache2() throws Exception
   {
      PojoCacheJmxWrapper wrapper = createWrapper(createConfiguration());
      wrapper.setRegisterPlainCache(false);
      wrapper.setRegisterInterceptors(true);

      wrapper.create();
      wrapper.start();

      registerWrapper(wrapper);

      assertTrue("Should be registered", mBeanServer.isRegistered(mBeanName));
      assertFalse("Plain cache should not be registered", mBeanServer.isRegistered(plainCacheMBeanName));
   }

//   public void testGetInternalLocation() throws Exception
//   {
//      PojoCacheJmxWrapperMBean wrapper = createWrapper(createConfiguration());
//      wrapper = registerWrapper(wrapper);
//      wrapper.create();
//      wrapper.start();
//
//      registerWrapper(wrapper);
//      PojoCache pc = wrapper.getPojoCache();
//
//      Person joe = new Person();
//      joe.setName("Joe");
//      joe.setAge(25);
//
//      pc.attach("/person/joe", joe);
//
//      assertEquals("Correct location", "/person/joe", wrapper.getInternalLocation(joe));
//   }

   //

   public void testDuplicateInvocation() throws Exception
   {
      PojoCacheJmxWrapperMBean cache = registerWrapper();
      cache.create();
      cache.start();
      cache.create();
      cache.start();

      cache.getPojoCache().attach("/a/b/c", null);

      cache.stop();
      cache.destroy();
      cache.stop();
      cache.destroy();
   }

   public void testFailedStart() throws Exception
   {
      PojoCacheJmxWrapper wrapper = new PojoCacheJmxWrapper(createCache(createConfiguration()));
      registerWrapper(wrapper);
      assertEquals("Correct state", CacheStatus.INSTANTIATED, wrapper.getCacheStatus());
      DisruptLifecycleListener listener = new DisruptLifecycleListener();
      wrapper.getPojoCache().getCache().addCacheListener(listener);
      wrapper.create();
      assertEquals("Correct state", CacheStatus.CREATED, wrapper.getCacheStatus());
      listener.setDisrupt(true);
      try
      {
         wrapper.start();
         fail("Listener did not prevent start");
      }
      catch (PojoCacheException good)
      {
      }

      assertEquals("Correct state", CacheStatus.FAILED, wrapper.getCacheStatus());

      listener.setDisrupt(false);

      wrapper.start();

      assertEquals("Correct state", CacheStatus.STARTED, wrapper.getCacheStatus());

      wrapper.getPojoCache().attach("/a/b/c", null);

      listener.setDisrupt(true);
      // need to re-add the listener since the failed start would have nullified the notifier.
      wrapper.getPojoCache().getCache().addCacheListener(listener);


      try
      {
         wrapper.stop();
         fail("Listener did not prevent stop");
      }
      catch (CacheException good)
      {
      }

      assertEquals("Correct state", CacheStatus.FAILED, wrapper.getCacheStatus());

      listener.setDisrupt(false);

      wrapper.stop();
      assertEquals("Correct state", CacheStatus.STOPPED, wrapper.getCacheStatus());
      wrapper.destroy();
      assertEquals("Correct state", CacheStatus.DESTROYED, wrapper.getCacheStatus());
   }


   @CacheListener
   public class DisruptLifecycleListener
   {
      private boolean disrupt;

      @CacheStarted
      public void cacheStarted(Event e)
      {
         if (disrupt) throw new IllegalStateException("I don't want to start");
      }

      @CacheStopped
      public void cacheStopped(Event e)
      {
         if (disrupt) throw new IllegalStateException("I don't want to stop");
      }

      public void setDisrupt(boolean disrupt)
      {
         this.disrupt = disrupt;
      }
   }
}
