/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.cache.pojo;

import static org.testng.AssertJUnit.assertEquals;
import static org.testng.AssertJUnit.assertNotNull;
import static org.testng.AssertJUnit.assertNull;

import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.cache.buddyreplication.NextMemberBuddyLocatorConfig;
import org.jboss.cache.config.BuddyReplicationConfig;
import org.jboss.cache.config.Configuration;
import org.jboss.cache.config.Configuration.CacheMode;
import org.jboss.cache.factories.UnitTestCacheConfigurationFactory;
import org.jboss.cache.pojo.test.Person;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

/**
 * A BuddyReplicatedTest.
 *
 * @author <a href="brian.stansberry@jboss.com">Brian Stansberry</a>
 * @version $Revision: 6999 $
 */
@Test(groups = {"functional"})
public class BuddyReplicationTest
{
   Log log = LogFactory.getLog(ReplicatedTest.class);
   PojoCache cache, cache1;

   @BeforeMethod(alwaysRun = true)
   protected void setUp() throws Exception
   {
      log.info("setUp() ....");
      boolean toStart = false;
      Configuration cfg1 = UnitTestCacheConfigurationFactory.createConfiguration(CacheMode.REPL_SYNC);
      addBuddyReplication(cfg1);
      cache = PojoCacheFactory.createCache(cfg1, toStart);
      cache.start();
      Configuration cfg2 = UnitTestCacheConfigurationFactory.createConfiguration(CacheMode.REPL_SYNC);
      addBuddyReplication(cfg2);
      cache1 = PojoCacheFactory.createCache(cfg2, toStart);
      cache1.start();

      // Enable gravitation
      cache.getThreadContext().setGravitationEnabled(true);
      cache1.getThreadContext().setGravitationEnabled(true);
   }

   @AfterMethod(alwaysRun = true)
   protected void tearDown() throws Exception
   {
      cache.stop();
      cache1.stop();
   }

   private void addBuddyReplication(Configuration cfg)
   {
      BuddyReplicationConfig brc = new BuddyReplicationConfig();
      brc.setAutoDataGravitation(false);
      brc.setBuddyPoolName("test");
      brc.setEnabled(true);
      NextMemberBuddyLocatorConfig blc = new NextMemberBuddyLocatorConfig();
      brc.setBuddyLocatorConfig(blc);

      cfg.setBuddyReplicationConfig(brc);
      cfg.setSyncCommitPhase(true);
   }

   private Person createPerson(String id, String name, int age)
   {
      Person p = new Person();
      p.setName(name);
      p.setAge(age);
      cache.attach(id, p);
      return p;
   }

   private Person getReplicatedPerson(String id) throws PojoCacheException
   {
      Person repl = (Person) cache1.find(id);
      assertNotNull("Person found at " + id, repl);
      return repl;
   }

   public void testSimple() throws Exception
   {
      log.info("testSimple() ....");
      Person ben = createPerson("/person/test1", "Ben Wang", 40);
      assertEquals("Ben Wang", ben.getName());
      Person repl = getReplicatedPerson("/person/test1");
      assertEquals("Ben Wang", repl.getName());
      cache.detach("/person/test1");
   }


   public void testDynamicRefSwapping() throws Exception
   {
      Person person = createPerson("/person/test3", "Joe", 32);
      try
      {
         person.setAge(30);
         List<String> med = person.getMedication();
         assertNull("Medication should be null ", med);
         person.setAge(61);
         med = person.getMedication();
         assertEquals("Medication ", (Object) "Lipitor", (Object) med.get(0));
         assertEquals("Medication on cache1 ", "Lipitor",
                 person.getMedication().get(0));

         person.setAge(71);
         assertEquals("Medication ", "Vioxx", med.get(1));
         Person repl = getReplicatedPerson("/person/test3");
         assertEquals("Medication on cache1 ", "Vioxx",
                      repl.getMedication().get(1));
         cache.detach("/person/test3");

      } catch (Exception e)
      {
         // should be thrown
      }
   }

   public void testTransient() throws Exception
   {
      log.info("testTransient() ....");
      Person ben = createPerson("/person/test1", "Ben Wang", 40);
      ben.setCurrentStatus("Idle");
      assertEquals("Cache 1 ", "Idle", ben.getCurrentStatus());
      Person repl = getReplicatedPerson("/person/test1");
      assertEquals("Cache 2 ", "Active",
                   repl.getCurrentStatus());
      cache.detach("/person/test1");
   }

   public void testModification() throws Exception
   {
      Person ben = createPerson("/person/test2", "Ben Wang", 40);
      ben.setName("Harald Gliebe");
      assertEquals(ben.getName(), "Harald Gliebe");
      Person repl = getReplicatedPerson("/person/test2");
      assertEquals("Harald Gliebe", repl.getName());
      cache.detach("/person/test2");
   }
}
