/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.cache.pojo.notification;

import static org.testng.AssertJUnit.assertEquals;
import static org.testng.AssertJUnit.assertSame;

import java.util.LinkedHashSet;
import java.util.Set;

import org.jboss.cache.pojo.PojoCache;
import org.jboss.cache.pojo.PojoCacheFactory;
import org.jboss.cache.pojo.notification.event.AttachedEvent;
import org.jboss.cache.pojo.notification.event.DetachedEvent;
import org.jboss.cache.pojo.notification.event.Event;
import org.jboss.cache.pojo.notification.event.SetModifiedEvent;
import org.jboss.cache.pojo.test.Person;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

//$Id: SetTest.java 7070 2008-11-04 00:37:28Z jason.greene@jboss.com $

/**
 * Tests set notifications
 *
 * @author Jason T. Greene
 */
@Test(groups = {"functional"})
public class SetTest
{
   protected PojoCache cache;
   protected Listener listener = new Listener();


   @BeforeMethod(alwaysRun = true)
   protected void setUp() throws Exception
   {
      String configFile = "META-INF/local-service.xml";
      boolean toStart = false;
      cache = PojoCacheFactory.createCache(configFile, toStart);
      cache.start();
      cache.addListener(listener);
   }

   @AfterMethod(alwaysRun = true)
   protected void tearDown() throws Exception
   {
      cache.stop();
      listener.clear();
   }

   private <T extends Event> T takeNotification(Class<T> clazz)
   {
      T notification = listener.take(clazz);
      verifyNotification(notification);

      return notification;
   }

   protected void verifyNotification(Event notification)
   {
      assertSame(cache, notification.getContext().getPojoCache());
      assertEquals(true, notification.isLocal());
   }

   @SuppressWarnings("unchecked")
   public void testSetAddNotification() throws Exception
   {
      final String test1 = "test1";
      final String test2 = "test2";

      Set<String> set = new LinkedHashSet<String>();
      set.add(test1);
      set.add(test2);
      cache.attach("a", set);
      set = (Set) cache.find("a");

      // String attach
      AttachedEvent attach = takeNotification(AttachedEvent.class);
      assertEquals(test1, attach.getSource());

      // Set add
      SetModifiedEvent modify = takeNotification(SetModifiedEvent.class);
      assertEquals(SetModifiedEvent.Operation.ADD, modify.getOperation());
      assertEquals(test1, modify.getValue());

      // String attach
      attach = takeNotification(AttachedEvent.class);
      assertEquals(test2, attach.getSource());

      // Set add
      modify = takeNotification(SetModifiedEvent.class);
      assertEquals(SetModifiedEvent.Operation.ADD, modify.getOperation());
      assertEquals(test2, modify.getValue());

      // Set Attach
      attach = takeNotification(AttachedEvent.class);
      assertEquals(set, attach.getSource());

      //cache.detach("a");
   }

   @SuppressWarnings("unchecked")
   public void testSetRemoveNotification() throws Exception
   {
      final String test1 = "test1";
      final String test2 = "test2";

      Set<String> set = new LinkedHashSet<String>();
      set.add(test1);
      set.add(test2);
      cache.attach("b", set);

      set = (Set<String>) cache.find("b");
      set.remove(test2);

      // String attach
      AttachedEvent attach = takeNotification(AttachedEvent.class);
      assertEquals(test1, attach.getSource());

      // Set add
      SetModifiedEvent modify = takeNotification(SetModifiedEvent.class);
      assertEquals(SetModifiedEvent.Operation.ADD, modify.getOperation());
      assertEquals(test1, modify.getValue());

      // String attach
      attach = takeNotification(AttachedEvent.class);
      assertEquals(test2, attach.getSource());

      // Set add
      modify = takeNotification(SetModifiedEvent.class);
      assertEquals(SetModifiedEvent.Operation.ADD, modify.getOperation());
      assertEquals(test2, modify.getValue());

      // Set Attach
      attach = takeNotification(AttachedEvent.class);
      assertEquals(set, attach.getSource());

      // Set remove
      modify = takeNotification(SetModifiedEvent.class);
      assertEquals(SetModifiedEvent.Operation.REMOVE, modify.getOperation());
      assertEquals(test2, modify.getValue());

      // String detach
      DetachedEvent detach = takeNotification(DetachedEvent.class);
      assertEquals(test2, detach.getSource());

      //cache.detach("b");
   }

   public void testObjectSetAdd() throws Exception
   {
      final String drumming = "druming";
      final String engineering = "engineering";

      Person test = new Person();
      test.setName("Joe");
      test.setAge(30);

      Set<String> set = new LinkedHashSet<String>();
      set.add(drumming);
      set.add(engineering);
      test.setSkills(set);
      cache.attach("c", test);

      // String attach
      AttachedEvent attach = takeNotification(AttachedEvent.class);
      assertEquals(drumming, attach.getSource());

      // List add
      SetModifiedEvent modify = takeNotification(SetModifiedEvent.class);
      assertEquals(SetModifiedEvent.Operation.ADD, modify.getOperation());
      assertEquals(drumming, modify.getValue());

      // String attach
      attach = takeNotification(AttachedEvent.class);
      assertEquals(engineering, attach.getSource());

      // List add
      modify = takeNotification(SetModifiedEvent.class);
      assertEquals(SetModifiedEvent.Operation.ADD, modify.getOperation());
      assertEquals(engineering, modify.getValue());

      // List Attach
      attach = takeNotification(AttachedEvent.class);
      assertEquals(test.getSkills(), attach.getSource());

      // Person Attach
      attach = takeNotification(AttachedEvent.class);
      assertEquals(test, attach.getSource());

      //cache.detach("c");
   }
}
