/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.cache.pojo.notification;

import static org.testng.AssertJUnit.assertEquals;
import static org.testng.AssertJUnit.assertSame;

import java.util.ArrayList;
import java.util.List;

import org.jboss.cache.pojo.PojoCache;
import org.jboss.cache.pojo.PojoCacheFactory;
import org.jboss.cache.pojo.notification.event.AttachedEvent;
import org.jboss.cache.pojo.notification.event.DetachedEvent;
import org.jboss.cache.pojo.notification.event.Event;
import org.jboss.cache.pojo.notification.event.ListModifiedEvent;
import org.jboss.cache.pojo.test.Person;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

//$Id: ListTest.java 6747 2008-09-17 23:24:20Z jason.greene@jboss.com $

/**
 * Tests list notifications
 *
 * @author Jason T. Greene
 */
@Test(groups = {"functional"})
public class ListTest
{
   protected PojoCache cache;
   protected Listener listener = new Listener();


   @BeforeMethod(alwaysRun = true)
   protected void setUp() throws Exception
   {
      String configFile = "META-INF/local-service.xml";
      boolean toStart = false;
      cache = PojoCacheFactory.createCache(configFile, toStart);
      cache.start();
      cache.addListener(listener);
   }

   @AfterMethod(alwaysRun = true)
   protected void tearDown() throws Exception
   {
      cache.stop();
      listener.clear();
   }

   private <T extends Event> T takeNotification(Class<T> clazz)
   {
      T notification = listener.take(clazz);
      verifyNotification(notification);

      return notification;
   }

   protected void verifyNotification(Event notification)
   {
      assertSame(cache, notification.getContext().getPojoCache());
      assertEquals(true, notification.isLocal());
   }

   @SuppressWarnings("unchecked")
   public void testListAddNotification() throws Exception
   {
      final String test1 = "test1";
      final String test2 = "test2";

      List<String> list = new ArrayList<String>();
      list.add(test1);
      list.add(test2);
      cache.attach("a", list);
      list = (List<String>) cache.find("a");

      // String attach
      AttachedEvent attach = (AttachedEvent) takeNotification(AttachedEvent.class);
      assertEquals(test1, attach.getSource());

      // List add
      ListModifiedEvent modify = takeNotification(ListModifiedEvent.class);
      assertEquals(ListModifiedEvent.Operation.ADD, modify.getOperation());
      assertEquals(test1, modify.getValue());
      assertEquals(0, modify.getIndex());

      // String attach
      attach = takeNotification(AttachedEvent.class);
      assertEquals(test2, attach.getSource());

      // List add
      modify = takeNotification(ListModifiedEvent.class);
      assertEquals(ListModifiedEvent.Operation.ADD, modify.getOperation());
      assertEquals(test2, modify.getValue());
      assertEquals(1, modify.getIndex());

      // List Attach
      attach = takeNotification(AttachedEvent.class);
      assertEquals(list, attach.getSource());
      //cache.detach("a");
   }

   @SuppressWarnings("unchecked")
   public void testListSetNotification() throws Exception
   {
      final String test1 = "test1";
      final String test2 = "test2";

      List<String> list = new ArrayList<String>();
      list.add(test1);
      cache.attach("b", list);
      list = (List<String>) cache.find("b");
      list.set(0, test2);

      // String attach
      AttachedEvent attach = takeNotification(AttachedEvent.class);
      assertEquals(test1, attach.getSource());

      // List add
      ListModifiedEvent modify = takeNotification(ListModifiedEvent.class);
      assertEquals(ListModifiedEvent.Operation.ADD, modify.getOperation());
      assertEquals(test1, modify.getValue());
      assertEquals(0, modify.getIndex());

      // List Attach
      attach = takeNotification(AttachedEvent.class);
      assertEquals(list, attach.getSource());

      // String detach
      DetachedEvent detach = takeNotification(DetachedEvent.class);
      assertEquals(test1, detach.getSource());

      // String attach
      attach = takeNotification(AttachedEvent.class);
      assertEquals(test2, attach.getSource());

      // List set
      modify = takeNotification(ListModifiedEvent.class);
      assertEquals(ListModifiedEvent.Operation.SET, modify.getOperation());
      assertEquals(test2, modify.getValue());
      assertEquals(0, modify.getIndex());
      //cache.detach("b");
   }

   @SuppressWarnings("unchecked")
   public void testListRemoveNotification() throws Exception
   {
      final String test1 = "test1";
      final String test2 = "test2";

      List<String> list = new ArrayList<String>();
      list.add(test1);
      list.add(test2);
      cache.attach("c", list);
      list = (List<String>) cache.find("c");
      list.remove(1);

      // String attach
      AttachedEvent attach = takeNotification(AttachedEvent.class);
      assertEquals(test1, attach.getSource());

      // List add
      ListModifiedEvent modify = takeNotification(ListModifiedEvent.class);
      assertEquals(ListModifiedEvent.Operation.ADD, modify.getOperation());
      assertEquals(test1, modify.getValue());
      assertEquals(0, modify.getIndex());

      // String attach
      attach = takeNotification(AttachedEvent.class);
      assertEquals(test2, attach.getSource());

      // List add
      modify = takeNotification(ListModifiedEvent.class);
      assertEquals(ListModifiedEvent.Operation.ADD, modify.getOperation());
      assertEquals(test2, modify.getValue());
      assertEquals(1, modify.getIndex());

      // List Attach
      attach = takeNotification(AttachedEvent.class);
      assertEquals(list, attach.getSource());

      // List remove
      modify = takeNotification(ListModifiedEvent.class);
      assertEquals(ListModifiedEvent.Operation.REMOVE, modify.getOperation());
      assertEquals(test2, modify.getValue());
      assertEquals(1, modify.getIndex());

      // String detach
      DetachedEvent detach = takeNotification(DetachedEvent.class);
      assertEquals(test2, detach.getSource());
      //cache.detach("c");
   }

   public void testObjectListAdd() throws Exception
   {
      final String english = "English";
      final String taiwanese = "Taiwanese";

      Person test = new Person();
      test.setName("Ben");
      test.setAge(10);

      ArrayList<String> list = new ArrayList<String>();

      list.add(english);
      list.add(taiwanese);
      test.setLanguages(list);

      cache.attach("d", test);

      // String attach
      AttachedEvent attach = takeNotification(AttachedEvent.class);
      assertEquals(english, attach.getSource());

      // List add
      ListModifiedEvent modify = takeNotification(ListModifiedEvent.class);
      assertEquals(ListModifiedEvent.Operation.ADD, modify.getOperation());
      assertEquals(english, modify.getValue());
      assertEquals(0, modify.getIndex());

      // String attach
      attach = takeNotification(AttachedEvent.class);
      assertEquals(taiwanese, attach.getSource());

      // List add
      modify = takeNotification(ListModifiedEvent.class);
      assertEquals(ListModifiedEvent.Operation.ADD, modify.getOperation());
      assertEquals(taiwanese, modify.getValue());
      assertEquals(1, modify.getIndex());

      // List Attach
      attach = takeNotification(AttachedEvent.class);
      assertEquals(test.getLanguages(), attach.getSource());

      // Person Attach
      attach = takeNotification(AttachedEvent.class);
      assertEquals(test, attach.getSource());

      //cache.detach("d");
   }


}
