/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 

package org.jboss.cache.search;

import org.hibernate.search.backend.TransactionContext;
import org.jboss.cache.notifications.event.NodeModifiedEvent;

import javax.transaction.Synchronization;
import javax.transaction.Transaction;

/**
 * This class implements the {@link org.hibernate.search.backend.TransactionContext} interface.  It
 * retrieves transaction context information from the {@link org.jboss.cache.notifications.event.NodeModifiedEvent} that gets passed in.
 * <p />
 * It is used by the {@link SearchableCoreListener} to pass transaction information to a Hibernate Search {@link org.hibernate.search.backend.Work} object.
 * <p />
 * @author Navin Surtani (<a href="mailto:nsurtani@redhat.com">nsurtani@redhat.com</a>)
 * @see SearchableCoreListener
 */
public class NodeModifiedTransactionContext implements TransactionContext
{

   NodeModifiedEvent event;

   /**
    * Creates a new instance of NodeModifiedTransactionContext.
    * <p />
    * @param event a NodeModifiedEvent to wrap.  Should not be null.
    * @throws NullPointerException if event is null.
    */
   public NodeModifiedTransactionContext(NodeModifiedEvent event)
   {
      if (event == null) throw new NullPointerException("event cannot be null");
      this.event = event;
   }

   /**
    * Returns a boolean value whether or not a transaction is in progress (JTA transaction and in this case *not*
    * an org.hibernate transaction).
    * @return true if a transaction is in progress, false otherwise.
    */
   public boolean isTransactionInProgress()
   {
      return (event.getTransaction() != null);
   }

   /**
    * Returns a JTA transaction.
    * @return a JTA transaction if one is available, or a null otherwise.
    * @see org.jboss.cache.notifications.event.NodeModifiedEvent#getTransaction()
    */

   public Object getTransactionIdentifier()
   {
      return event.getTransaction();
   }

   /**
    * Registers the synchronization passed in as a parameter with the ongoing transaction.
    * <p />
    * If there is no ongoing transaction, then this method will do nothing and simply return.
    * <p />
    * @param synchronization synchronization to register.  Must not be null.
    * @throws NullPointerException if the synchronization is null.
    */
   public void registerSynchronization(Synchronization synchronization)
   {
      if (synchronization == null) throw new NullPointerException("Synchronization passed in is null!");

      Transaction transaction = event.getTransaction();
      if (transaction != null)
      {
         try
         {
            transaction.registerSynchronization(synchronization);
         }
         catch (Exception e)
         {
            throw new RuntimeException(e);
         }
      }
   }
}
