/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.cache.search;

import org.jboss.cache.Fqn;

/**
 *
 * This class is used to get fqns, keys and documentId's by calling methods on {@link org.jboss.cache.search.Transformer}
 * <p/>
 * @author Navin Surtani (<a href="mailto:nsurtani@redhat.com">nsurtani@redhat.com</a>)
 */
public class CacheEntityId
{
   Fqn fqn;
   String key;
   String documentId;

   public CacheEntityId(String documentId)
   {
      if(documentId == null) throw new NullPointerException("documentId is null");
      this.documentId = documentId;
   }

   public CacheEntityId(Fqn fqn, String key)
   {
      if(fqn == null) throw new NullPointerException("Fqn is null");
      if(key == null) throw new NullPointerException("Key is null");
      this.fqn = fqn;
      this.key = key;
   }

   /**
    * Gets the Fqn from the instance of CacheEntityId.
    *
    * @return Fqn from the instance of CacheEntityId.
    */

   public Fqn getFqn()
   {
      if (fqn != null) return fqn;
      if (documentId != null)
      {
         fqn = Transformer.getFqn(documentId);
         return fqn;
      }

      if(documentId == null)
      throw new IllegalArgumentException("docId is null");

      throw new IllegalArgumentException("Fqn is null");
   }

   /**
    * Gets the key from the instance of CacheEntityId.
    *
    * @return Key from the instance of CacheEntityId.
    */

   public String getKey()
   {
      if (key != null) return key;
      if (documentId != null)
      {
         key = Transformer.getKey(documentId);
         return key;
      }

      throw new IllegalArgumentException("At least key or documentId must be set to call this method");
   }

   /**
    * Gets a documentId String from an Fqn and key combination.
    *
    * @return documentId String.
    */


   public String getDocumentId() throws InvalidKeyException
   {
      if (key == null || fqn == null)
      {
         throw new IllegalArgumentException("Either your key or fqn is null. Please check again.");
      }

      return Transformer.generateId(fqn, key);
   }
}
