/*
 * JBoss, Home of Professional Open Source
 * Copyright ${year}, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 

package org.jboss.cache.search;

import org.jboss.cache.Fqn;

/**
*
* This class is one that does all the 'conversion' work between JBossCache and Hibernate Search. This is where
* users can switch from Fqn and key to a documentId and vice versa.
*
* If the Fqn is in the form /a/b/c and the key - which has to be a String - is keystring; the documentId - which is also
* a String - will be "Fqn=[/a/b/c]Key=[keystring]"
*
* @author Navin Surtani (<a href="mailto:nsurtani@redhat.com">nsurtani@redhat.com</a>)
*/
public class Transformer
{
   /**
    * Takes in the documentId string from the user and will return the key from the Fqn, key combination.
    *
    *
    * @param docId - for the keystring to be obtained
    * @return keystring.
    */

   public static String getKey(String docId)
   {
      //docID comes in the format "Fqn=[/a/b/c]Key=[key]"

      // This will be the index of the first time the sub-String "key=[" occurs within the whole String.
      int startIndex = docId.indexOf("Key=[");

      //The index of the endIndex of the key sequence so we know when to cut out. The startIndex integer is also passed in so that
      // there will not be an error later of endIndex < startIndex because this char occurs earlier when the Fqn is stated.
      int endIndex = docId.indexOf("]", startIndex);

      //Make the startIndex index point at the first char in the key sequence.
      startIndex += 5;

      //The resultant key that will be returned.
      String key = docId.substring(startIndex, endIndex);

      return key;
   }

   /**
    * Takes in the documentId string from the user and will return the Fqn from the Fqn, key combination.
    *
    * @param docId - for the Fqn to be obtained
    * @return Fqn from the documentId.
    */


   public static Fqn getFqn(String docId)
   {
      //docId comes in the format "Fqn=[/a/b/c]Key=[key]"


      // This will be the index of the first time the sub-String "Fqn=[" occurs within the whole String.
      //Adding 1 so that the index being pointed at will be the first character in the Fqn sequence.
      int startIndex = docId.indexOf("[") + 1;

      //The endIndex of the Fqn sequence so that we know when to cut out the sub-String.
      int endIndex = docId.indexOf("]");

      String fqnString = docId.substring(startIndex, endIndex);

      Fqn fqn = Fqn.fromString(fqnString);
      return fqn;
   }

   /**
    * Generates the documentId from an Fqn, key combination.
    *
    *
    * @param fqn - standard call Fqn.fromString()
    * @param key - cannot be an object.
    * @return documentId String to be given to Lucene.
    * @throws InvalidFqnException
    */

   public static String generateId(Fqn fqn, String key) throws InvalidFqnException, InvalidKeyException
   {
      if (key == null) throw new NullPointerException("Key passed in cannot be null!");

      if (key.startsWith("Fqn")) throw new InvalidKeyException("Invalid key passed in - 'Fqn[' is not allowed");
      if (key.startsWith("]Key=[")) throw new InvalidKeyException("Invalid key passed in - ']Key=[' is not allowed");

      if (fqn == null) throw new NullPointerException("Fqn cannot be null");

      if(fqn.toString().startsWith("/Fqn")) throw new InvalidFqnException("Invalid Fqn passed in.");
      if (fqn.toString().startsWith("/]Key")) throw new InvalidFqnException("Invalid Fqn passed in.");
      
      StringBuilder sb = new StringBuilder();
      sb.append("Fqn=[");
      sb.append(fqn);
      sb.append("]Key=[");
      sb.append(key);
      sb.append("]");

      return sb.toString();

   }


}
