/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.aspects.remoting;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import org.jboss.aop.Dispatcher;
import org.jboss.aop.advice.Interceptor;
import org.jboss.aop.joinpoint.MethodInvocation;
import org.jboss.aop.util.MethodHashing;
import org.jboss.aop.util.PayloadKey;
import org.jboss.ha.client.loadbalance.LoadBalancePolicy;
import org.jboss.remoting.InvokerLocator;

/**
 * InvocationHandler that generates an aop {@link MethodInvocation} and 
 * populates its metadata with a target URI as well as various pieces of
 * clustering metadata.
 *
 * @author Brian Stansberry, based on work by Bill Burke
 */
public class ClusteredPojiProxy implements java.io.Serializable, InvocationHandler
{
   private static final long serialVersionUID = 8054816523858555978L;

   private Object oid;
   private InvokerLocator uri;
   private Interceptor[] interceptors;
   private FamilyWrapper family;
   private LoadBalancePolicy lbPolicy;
   private String partitionName;
   private Object originTarget;

   public ClusteredPojiProxy(Object oid, InvokerLocator uri, Interceptor[] interceptors,
                             FamilyWrapper family, LoadBalancePolicy lb, 
                             String partitionName, Object originTarget)
   {
      this.oid = oid;
      this.uri = uri;
      this.interceptors = interceptors;
      this.family = family;
      this.lbPolicy = lb;
      this.partitionName = partitionName;
      this.originTarget = originTarget;
   }

   public Object invoke(Object proxy, Method method, Object[] args) throws Throwable
   {
      long hash = MethodHashing.calculateHash(method);
      MethodInvocation sri = new MethodInvocation(interceptors, hash, method, method, null);
      sri.setArguments(args);
      sri.getMetaData().addMetaData(ClusterConstants.CLUSTERED_REMOTING, ClusterConstants.CLUSTER_FAMILY_WRAPPER, family, PayloadKey.TRANSIENT);
      sri.getMetaData().addMetaData(ClusterConstants.CLUSTERED_REMOTING, ClusterConstants.LOADBALANCE_POLICY, lbPolicy, PayloadKey.TRANSIENT);
      sri.getMetaData().addMetaData(ClusterConstants.CLUSTERED_REMOTING, ClusterConstants.PARTITION_NAME, partitionName, PayloadKey.TRANSIENT);
      if (originTarget != null)
         sri.getMetaData().addMetaData(ClusterConstants.CLUSTERED_REMOTING, ClusterConstants.HA_TARGET, originTarget, PayloadKey.TRANSIENT);
      sri.getMetaData().addMetaData(Dispatcher.DISPATCHER, Dispatcher.OID, oid, PayloadKey.AS_IS);
      sri.getMetaData().addMetaData(InvokeRemoteInterceptor.REMOTING, InvokeRemoteInterceptor.INVOKER_LOCATOR, uri, PayloadKey.AS_IS);
      sri.getMetaData().addMetaData(InvokeRemoteInterceptor.REMOTING, InvokeRemoteInterceptor.SUBSYSTEM, "AOP", PayloadKey.AS_IS);
      return sri.invokeNext();
   }
}
