/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.ha.framework.server;

import java.util.Hashtable;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NameNotFoundException;
import javax.naming.NamingException;

import org.jboss.ha.framework.interfaces.HAPartition;
import org.jboss.logging.Logger;

/**
 * Service Locator utility for locating an {@link HAPartition}. Maintains
 * an internal registry of partitions, and if it cannot find a partition there,
 * will attempt to find it in JNDI using a standard naming pattern.
 * 
 * @author <a href="brian.stansberry@jboss.com">Brian Stansberry</a>
 * @version $Revision: 1.1 $
 */
public class HAPartitionLocator
{
   private static final Logger log = Logger.getLogger(HAPartitionLocator.class);
   
   private static HAPartitionLocator sharedInstance = new HAPartitionLocator();
   
   /** Name of the standard JNDI context under which HAPartitions are bound */
   public static final String JNDI_CONTEXT_NAME = "/HAPartition/";
   
   private Map<String, HAPartition> registeredPartitions = new ConcurrentHashMap<String, HAPartition>();
   
   /**
    * Gets the 
    * @return the shared HAPartitionLocator; will not be <code>null</code>
    */
   public static HAPartitionLocator getHAPartitionLocator()
   {
      return sharedInstance;
   }
   
   /**
    * Appends the partition name to {@link #JNDI_CONTEXT_NAME}; e.g.
    * <code>/HAPartition/DefaultPartition</code>.
    * 
    * @param partitionName the name of the partition
    * @return a string representing the standard name under which a 
    *         partition is bound in JNDI.
    */
   public static final String getStandardJndiBinding(String partitionName)
   {
      return JNDI_CONTEXT_NAME + partitionName;
   }
   
   /**
    * Allows replacement of the default implementation.
    * 
    * @param shared the locator to use. Cannot be <code>null</code>.
    */
   protected static void setSharedInstance(HAPartitionLocator shared)
   {
      if (shared == null)
         throw new IllegalArgumentException("shared cannot be null");
      
      sharedInstance = shared;
   }
   
   /**
    * Allow subclasses to create a new HAPartitionLocator.
    */
   protected HAPartitionLocator()
   {      
   }
   
   /**
    * Locates and returns the {@link HAPartition} whose partition name matches
    * the given <code>partitionName</code>.
    * 
    * @param partitionName  the name of the partition. Cannot be <code>null</code>.
    * @param jndiProperties any naming properties to pass to new InitialContext() 
    *                       if JNDI lookup is needed.
    * @return the partition. Will not return <code>null</code>
    * 
    * @throws IllegalStateException if no matching partition can be found
    */
   public HAPartition getHAPartition(String partitionName, Hashtable jndiProperties)
   {
      HAPartition partition = findInRegistry(partitionName);
      if (partition == null)
      {
         try
         {
            partition = findInJndi(partitionName, jndiProperties);
         }
         catch (NamingException e)
         {
            log.error("Problem finding partition " + partitionName + " in JNDI", e);        
         }
      }
      
      if (partition == null)
         throw new IllegalStateException("Partition " + partitionName + " not found");
      
      return partition;
   }
   
   /**
    * Register a partition with this locator.
    * 
    * @param partition the partition
    */
   public void registerHAPartition(HAPartition partition)
   {
      if (partition != null)
         registeredPartitions.put(partition.getPartitionName(), partition);
   }
   
   /**
    * Deregister a partition from this locator.
    * 
    * @param partition the partition
    */
   public void deregisterHAPartition(HAPartition partition)
   {
      if (partition != null)
         registeredPartitions.remove(partition.getPartitionName());
   }
   
   /**
    * Find the given partition in the local registry.
    * 
    * @param partitionName the name of the partition. Cannot be <code>null</code>.
    * @return the partition, or <code>null</code>
    */
   protected HAPartition findInRegistry(String partitionName)
   {
      return registeredPartitions.get(partitionName);
   }
   
   /**
    * Find the given partition in JNDI under a 
    * {@link #getStandardJndiBinding(String) standard binding}.
    * 
    * @param partitionName the name of the partition. Cannot be <code>null</code>.
    * @return the partition, or <code>null</code>
    * 
    * @throws NamingException if there is a problem with the naming context. Will
    *                         not throw NameNotFoundException; if not found, will
    *                         return <code>null</code>.
    *                         
    * @see #getStandardJndiBinding(String)
    */
   protected HAPartition findInJndi(String partitionName, Hashtable jndiProperties) throws NamingException
   {
      try
      {
         Context ctx = new InitialContext(jndiProperties);
         return (HAPartition) ctx.lookup(getStandardJndiBinding(partitionName));
      }
      catch (NameNotFoundException e)
      {
         // just not there
         return null;
      }     
      
   }
}
