/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.web.tomcat.service.session.distributedcache.impl.jbc;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.jboss.cache.Cache;
import org.jboss.cache.CacheException;
import org.jboss.cache.Fqn;
import org.jboss.cache.Node;
import org.jboss.web.tomcat.service.session.distributedcache.spi.ClusteringNotSupportedException;
import org.jboss.web.tomcat.service.session.distributedcache.spi.LocalDistributableSessionManager;
import org.jboss.web.tomcat.service.session.distributedcache.spi.OutgoingAttributeGranularitySessionData;

/**
 * DistributedCacheManager impl for ReplicationGranularity.ATTRIBUTE.
 */
public class AttributeBasedJBossCacheService extends AbstractJBossCacheService<OutgoingAttributeGranularitySessionData>
{   
   public AttributeBasedJBossCacheService(LocalDistributableSessionManager localManager) throws ClusteringNotSupportedException
   {
      super(localManager);
   }
   
   public AttributeBasedJBossCacheService(LocalDistributableSessionManager localManager, Cache<Object, Object> cache)
   {
      super(localManager, cache);
   }

   public boolean getSupportsAttributeOperations()
   {
      return true;
   }

   public Object getAttribute(String realId, String key)
   {
      Fqn<String> fqn = getSessionFqn(combinedPath_, realId);
      return unmarshall(realId, key, cacheWrapper_.get(fqn, key));
   }

   public void putAttribute(String realId, String key, Object value)
   {
      Fqn<String> fqn = getSessionFqn(combinedPath_, realId);
      cacheWrapper_.put(fqn, key, getMarshalledValue(value));
   }

   public void putAttribute(String realId, Map<String, Object> map)
   {
      // Duplicate the map with marshalled values
      Map<Object, Object> marshalled = new HashMap<Object, Object>(map.size());
      for (Map.Entry<String, Object> entry : map.entrySet())
      {
         marshalled.put(entry.getKey(), getMarshalledValue(entry.getValue()));
      }
      
      Fqn<String> fqn = getSessionFqn(combinedPath_, realId);
      cacheWrapper_.put(fqn, marshalled);
      
   }

   public Object removeAttribute(String realId, String key)
   {
      Fqn<String> fqn = getSessionFqn(combinedPath_, realId);
      if (log_.isTraceEnabled())
      {
         log_.trace("Removing attribute from distributed store for session " + 
               Util.maskId(realId) +". Parent Fqn: " + fqn.getParent() + 
               " key: " + key);
      }
      return unmarshall(realId, key, cacheWrapper_.remove(fqn, key));
   }

   public void removeAttributeLocal(String realId, String key)
   {
      Fqn<String> fqn = getSessionFqn(combinedPath_, realId);
      if (log_.isTraceEnabled())
      {
         log_.trace("Removing attribute from my own distributed store only for session " + 
               Util.maskId(realId) +". Parent Fqn: " + fqn.getParent() + 
               " key: " + key);
      }
      cacheWrapper_.removeLocal(fqn, key);
   }

   /**
    * Obtain the keys associated with this fqn. Note that it is not the fqn children.
    *
    */
   @SuppressWarnings("unchecked")
   public Set<String> getAttributeKeys(String realId)
   {
      Set keys = null;
      Fqn<String> fqn = getSessionFqn(combinedPath_, realId);
      try
      {
         Node<Object, Object> node = getCache().getRoot().getChild(fqn);
         if (node != null)
         {
            keys = node.getKeys();
            keys.removeAll(INTERNAL_KEYS);
         }
      }
      catch (CacheException e)
      {
         log_.error("getAttributeKeys(): Exception getting keys for session " + Util.maskId(realId), e);
      }
      
      return keys;
   }

   /**
    * Return all attributes associated with this session id.
    * 
    * @param realId the session id with any jvmRoute removed
    * @return the attributes, or any empty Map if none are found.
    */
   public Map<String, Object> getAttributes(String realId)
   {
      if (realId == null || realId.length() == 0) 
      {
         @SuppressWarnings("unchecked")
         Map<String, Object> empty = Collections.EMPTY_MAP;
         return empty;
      }
      
      Fqn<String> fqn = getSessionFqn(combinedPath_, realId);
      
      Node<Object, Object> node = getCache().getRoot().getChild(fqn);
      Map<Object, Object> rawData = node.getData();
      
      return getSessionAttributes(realId, rawData);
   }
   
   /**
    * Returns the session attributes, possibly using the passed in 
    * <code>distributedCacheData</code> as a source.
    * 
    * <strong>Note:</strong> This operation may alter the contents of the 
    * passed in map. If this is unacceptable, pass in a defensive copy.
    */
   protected Map<String, Object> getSessionAttributes(String realId, Map<Object, Object> distributedCacheData)
   {
      Map<String, Object> attrs = new HashMap<String, Object>();         
      for (Map.Entry<Object, Object> entry : distributedCacheData.entrySet())
      {
         if (entry.getKey() instanceof String)
         {
            String key = (String) entry.getKey();
            attrs.put(key, unmarshall(realId, key, entry.getValue()));
         }                
      }
      
      return attrs;
   }

   @Override
   protected void storeSessionAttributes(Map<Object, Object> dataMap,
         OutgoingAttributeGranularitySessionData sessionData)
   {
      Fqn<String> fqn = null;
      Map<String, Object> map = sessionData.getModifiedSessionAttributes();
      if (map != null)
      {
         // Duplicate the map with marshalled values
         Map<Object, Object> marshalled = new HashMap<Object, Object>(map.size());
         for (Map.Entry<String, Object> entry : map.entrySet())
         {
            marshalled.put(entry.getKey(), getMarshalledValue(entry.getValue()));
         }
         fqn = getSessionFqn(combinedPath_, sessionData.getRealId());
         cacheWrapper_.put(fqn, marshalled);
      }
      
      Set<String> removed = sessionData.getRemovedSessionAttributes();
      if (removed != null)
      {
         if (fqn == null)
         {
            fqn = getSessionFqn(combinedPath_, sessionData.getRealId());
         }
         for (String key : removed)
         {
            cacheWrapper_.remove(fqn, key);
         }
      }
   }
   
   /**
    * Utility to catch checked unmarshalling exceptions and rethrow as 
    * RuntimeException with a context-appropriate message.
    * 
    * @param realId the id of session associated with toUnmarshall
    * @param key the attribute key associated with toUnmarshall
    * @param toUnmarshall object to unmarshall
    * 
    * @return the unmarshalled object
    * 
    * @throws RuntimeException if any exception occurs
    */
   private Object unmarshall(String realId, String key, Object toUnmarshall)
   {
      try
      {
         return getUnMarshalledValue(toUnmarshall);
      }
      catch (RuntimeException e)
      {
         throw e;
      }
      catch (Exception e)
      {
         throw new RuntimeException("Caught exception unmarshalling attribute " + key + " for session " +  Util.maskId(realId), e);
      }
      
   }

}
