/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.web.tomcat.service.session.distributedcache.impl;

import java.io.File;

import javax.management.MBeanServer;
import javax.management.MBeanServerInvocationHandler;
import javax.management.ObjectName;

import org.jboss.cache.Cache;
import org.jboss.cache.DefaultCacheFactory;
import org.jboss.cache.config.Configuration;
import org.jboss.cache.jmx.CacheJmxWrapper;
import org.jboss.metadata.web.jboss.ReplicationGranularity;
import org.jboss.web.tomcat.service.session.distributedcache.impl.jbc.AttributeBasedJBossCacheService;
import org.jboss.web.tomcat.service.session.distributedcache.impl.jbc.SessionBasedJBossCacheService;
import org.jboss.web.tomcat.service.session.distributedcache.impl.jbc.Util;
import org.jboss.web.tomcat.service.session.distributedcache.spi.ClusteringNotSupportedException;
import org.jboss.web.tomcat.service.session.distributedcache.spi.DistributedCacheManager;
import org.jboss.web.tomcat.service.session.distributedcache.spi.LocalDistributableSessionManager;
import org.jboss.web.tomcat.service.session.distributedcache.spi.TomcatClusterConfig;
import org.jboss.web.tomcat.service.session.distributedcache.spi.TomcatClusterDistributedCacheManagerFactory;

/**
 * @author Brian Stansberry
 *
 */
public class DistributedCacheManagerFactoryImpl implements TomcatClusterDistributedCacheManagerFactory
{
   public static final String DEFAULT_CLUSTER_NAME = "Tomcat-Cluster";

   /** TreeCache's isolation level */
   public static final String DEFAULT_ISOLATION_LEVEL = "REPEATABLE_READ";
   
   /** TreeCache's cache mode */
   public static final String DEFAULT_CACHE_MODE = "REPL_ASYNC";
   
   /** TreeCache's lock aquisition timeout */
   public static final long DEFAULT_LOCK_TIMEOUT = 15000;
   
   /** TransactionManagerLookup implementation that the TreeCache should use. */
   @SuppressWarnings("deprecation")
   public static final String DEFAULT_TM_LOOKUP = 
      org.jboss.cache.transaction.BatchModeTransactionManagerLookup.class.getName();
   
   private TomcatClusterConfig tomcatConfig;
   private boolean cacheLocal = false;
   private Cache<Object, Object> plainCache;
   
   @SuppressWarnings("unchecked")
   public DistributedCacheManager getDistributedCacheManager(LocalDistributableSessionManager localManager)
         throws ClusteringNotSupportedException
   {
      ReplicationGranularity granularity = Util.getReplicationGranularity(localManager);
      switch(granularity)
      {
         case SESSION:
            return plainCache == null? new SessionBasedJBossCacheService(localManager) : new SessionBasedJBossCacheService(localManager, plainCache);
         case ATTRIBUTE:
            return plainCache == null? new AttributeBasedJBossCacheService(localManager) : new AttributeBasedJBossCacheService(localManager, plainCache);
         default:
            throw new IllegalStateException("Unknown ReplicationGranularity " + granularity);
      }
   }

   public TomcatClusterConfig getTomcatClusterConfig()
   {
      return tomcatConfig;
   }

   public void setTomcatClusterConfig(TomcatClusterConfig clusterConfig)
   {
      this.tomcatConfig = clusterConfig;
   }

   public void start() throws Exception
   {
      if (tomcatConfig != null)
      {
         initializeCache();
      }      
   }

   public void stop() throws Exception
   {
      if (plainCache != null)
      {
         plainCache.stop();
         plainCache.destroy();

         if (cacheLocal && tomcatConfig.getMBeanServer() != null && tomcatConfig.getCacheObjectName() != null)
         {
            tomcatConfig.getMBeanServer().unregisterMBean(new ObjectName(tomcatConfig.getCacheObjectName()));
         }
      }

   }

   public Cache<Object, Object> getPlainCache()
   {
      return plainCache;
   }

   /** 
    * Hook for test fixtures to inject a Cache, which if present will
    * be used to create the DistributedCacheManager in preference to any
    * passed <code>cacheConfigName</code>.
    */
   public void setPlainCache(Cache<Object, Object> plainCache)
   {
      this.plainCache = plainCache;
   }
   
   /**
    * Convenience method for test fixtures to clear any injected cache.
    */
   public void clearCaches()
   {
      this.plainCache = null;
   }
   
   /**
    * Gets our Cache, either from a local reference or the JMX
    * server.  If one is not found, creates and configures it.
    */
   private void initializeCache() throws Exception
   {
      if (plainCache == null) {
         
         CacheJmxWrapper pcWrapper = null;
         MBeanServer server = tomcatConfig.getMBeanServer();
         String cfgName = tomcatConfig.getCacheObjectName();
         ObjectName objName = cfgName == null ? null : new ObjectName(tomcatConfig.getCacheObjectName());
         if (server != null && objName != null && server.isRegistered(objName))
         {
            // Get a proxy to the existing TreeCache
            pcWrapper = ((CacheJmxWrapper)
                           MBeanServerInvocationHandler.newProxyInstance(server, objName, CacheJmxWrapper.class, false));
         }
         else
         {
            // See if there is an XML descriptor file to configure the cache
            File configFile = tomcatConfig.getCacheConfigFile();
            String clusterName = tomcatConfig.getClusterName();
            
            if (configFile != null)
            {
               pcWrapper = new CacheJmxWrapper(DefaultCacheFactory.getInstance().createCache(configFile.getAbsolutePath(), false));
                              
               if (clusterName != null)
               {
                  // Override the XML config with the name provided in
                  // server.xml.  Method setClusterName is specified in the
                  // Cluster interface, otherwise we would not do this
                  pcWrapper.getCache().getConfiguration().setClusterName(clusterName);
               }
            }
            else
            {
               // User did not try to configure the cache.
               // Configure it using defaults.  Only exception
               // is the clusterName, which user can specify in server.xml.
               Configuration config = new Configuration();
               String channelName = (clusterName == null) ? DEFAULT_CLUSTER_NAME
                                                          : clusterName;
               config.setClusterName(channelName);
               config.setIsolationLevel(DEFAULT_ISOLATION_LEVEL);
               config.setCacheMode(DEFAULT_CACHE_MODE);
               config.setLockAcquisitionTimeout(DEFAULT_LOCK_TIMEOUT);
               config.setTransactionManagerLookupClass(DEFAULT_TM_LOOKUP);
               pcWrapper = new CacheJmxWrapper(DefaultCacheFactory.getInstance().createCache(config, false));

            }
            
            if (server != null && objName != null)
            {
               server.registerMBean(pcWrapper, objName);
            }
            
            cacheLocal = true;
         }
         setPlainCache(pcWrapper.getCache());
      }
   }
   

}
