/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.web.tomcat.service.session.distributedcache.impl;

import java.io.IOException;
import java.io.Serializable;
import java.security.AccessController;

import org.jboss.ha.framework.interfaces.ObjectStreamSource;
import org.jboss.ha.framework.server.MarshalledValueHelper;
import org.jboss.ha.framework.server.SimpleCachableMarshalledValue;
import org.jboss.util.loading.ContextClassLoaderSwitcher;
import org.jboss.web.tomcat.service.session.distributedcache.spi.LocalDistributableSessionManager;
import org.jboss.web.tomcat.service.session.distributedcache.spi.SessionAttributeMarshaller;

/**
 * Session attribute marshaller that marshals attribute values using a {@link SimpleCachableMarshalledValue}.
 * @author Paul Ferraro
 */
public class SessionAttributeMarshallerImpl implements SessionAttributeMarshaller
{
   @SuppressWarnings("unchecked")
   // Need to cast since ContextClassLoaderSwitcher.NewInstance does not generically implement PrivilegedAction<ContextClassLoaderSwitcher>
   private final ContextClassLoaderSwitcher switcher = (ContextClassLoaderSwitcher) AccessController.doPrivileged(ContextClassLoaderSwitcher.INSTANTIATOR);
   
   private final LocalDistributableSessionManager manager;
   private final ObjectStreamSource source;
   
   public SessionAttributeMarshallerImpl(LocalDistributableSessionManager manager, ObjectStreamSource source)
   {
      this.manager = manager;
      this.source = source;
   }
   
   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.SessionAttributeMarshaller#marshal(java.lang.Object)
    */
   @Override
   public Object marshal(Object value)
   {
      if ((value == null) || MarshalledValueHelper.isTypeExcluded(value.getClass())) return value;

      if (!(value instanceof Serializable))
      {
         throw new IllegalArgumentException(String.format("%s does not implement %s", value, Serializable.class.getName()));
      }
      
      return new SimpleCachableMarshalledValue((Serializable) value, this.source, true);
   }

   /**
    * {@inheritDoc}
    * @see org.jboss.web.tomcat.service.session.distributedcache.spi.SessionAttributeMarshaller#unmarshal(java.lang.Object)
    */
   @Override
   public Object unmarshal(Object object) throws IOException, ClassNotFoundException
   {
      if ((object == null) || !(object instanceof SimpleCachableMarshalledValue)) return object;

      SimpleCachableMarshalledValue value = (SimpleCachableMarshalledValue) object;
      
      // Swap in/out the class loader for this web app. Needed only for unmarshalling.
      ContextClassLoaderSwitcher.SwitchContext switchContext = this.switcher.getSwitchContext(this.manager.getApplicationClassLoader());
      
      try
      {
         value.setObjectStreamSource(this.source);
         
         return value.get();
      }
      finally
      {
         switchContext.reset();
      }
   }
}
