/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat, Inc., and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.gwt.elemento.template;

import com.google.gwt.user.client.ui.IsWidget;
import com.google.gwt.user.client.ui.Widget;
import elemental2.dom.Attr;
import elemental2.dom.DomGlobal;
import elemental2.dom.HTMLElement;
import elemental2.dom.NamedNodeMap;
import elemental2.dom.Node;
import elemental2.dom.NodeFilter;
import elemental2.dom.TreeWalker;
import org.jboss.gwt.elemento.core.Elements;
import org.jboss.gwt.elemento.core.IsElement;

/**
 * Static helper methods used from code generated by {@code @Templated} annotation processors. You should not need to
 * call any of these methods manually.
 *
 * @author Harald Pehl
 */
public final class TemplateUtil {

    @FunctionalInterface
    private interface SelectorFunction {

        HTMLElement select(HTMLElement context, String identifier);
    }


    private static SelectorFunction DATA_ELEMENT = (context, identifier) ->
            ((HTMLElement) context.querySelector("[data-element=" + identifier + "]"));

    private TemplateUtil() {}


    // ------------------------------------------------------ HTMLElement methods

    public static HTMLElement resolveElement(HTMLElement context, String identifier) {
        return DATA_ELEMENT.select(context, identifier);
    }

    @SuppressWarnings("unchecked")
    public static <E extends HTMLElement> E resolveElementAs(HTMLElement context, String identifier) {
        return (E) DATA_ELEMENT.select(context, identifier);
    }

    public static void replaceElement(HTMLElement context, String identifier, HTMLElement newElement) {
        if (newElement == null) {
            throw new NullPointerException("New element must not be null in TemplateUtils.replaceElement()");
        }
        HTMLElement oldElement = resolveElement(context, identifier);
        if (oldElement != null && oldElement.parentNode != null) {
            oldElement.parentNode.replaceChild(newElement, oldElement);
        }
    }


    // ------------------------------------------------------ IsElement / (Is)Widget methods

    public static void replaceIsElement(HTMLElement context, String identifier, IsElement newElement) {
        replaceElement(context, identifier, newElement.asElement());
    }

    public static void replaceWidget(HTMLElement context, String identifier, Widget newWidget) {
        replaceElement(context, identifier, Elements.asElement(newWidget));
    }

    public static void replaceIsWidget(HTMLElement context, String identifier, IsWidget newWidget) {
        replaceElement(context, identifier, Elements.asElement(newWidget));
    }


    // ------------------------------------------------------ handlebars

    public static void replaceHandlebar(HTMLElement context, String expression, String value) {
        replaceNestedHandlebarInText(context, expression, value);
        replaceNestedHandlebarInAttributes(context, expression, value);
        // The call above does not catch the attributes in 'context', we need to replace them explicitly.
        replaceHandlebarInAttributes(context, expression, value);
    }

    private static void replaceNestedHandlebarInText(HTMLElement context, String expression, String value) {
        TreeWalker treeWalker = DomGlobal.document.createTreeWalker(context, NodeFilter.SHOW_TEXT, node -> {
            if (node.nodeValue != null && node.nodeValue.contains(expression)) {
                return NodeFilter.FILTER_ACCEPT;
            }
            return NodeFilter.FILTER_SKIP;
        }, false);

        while (treeWalker.nextNode() != null) {
            treeWalker.getCurrentNode().nodeValue = treeWalker.getCurrentNode().nodeValue.replace(expression, value);
        }
    }

    private static void replaceNestedHandlebarInAttributes(HTMLElement context, String expression, String value) {
        TreeWalker treeWalker = DomGlobal.document.createTreeWalker(context, NodeFilter.SHOW_ELEMENT, null, false);
        while (treeWalker.nextNode() != null) {
            if (treeWalker.getCurrentNode() instanceof HTMLElement) {
                replaceHandlebarInAttributes((HTMLElement) treeWalker.getCurrentNode(), expression, value);
            }
        }
    }

    private static void replaceHandlebarInAttributes(HTMLElement context, String expression, String value) {
        NamedNodeMap<Attr> attributes = context.attributes;
        for (int i = 0; i < attributes.getLength(); i++) {
            Node attribute = attributes.item(i);
            String currentValue = attribute.nodeValue;
            if (currentValue.contains(expression)) {
                attribute.nodeValue = currentValue.replace(expression, value);
            }
        }
    }
}
