/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.deployers.spi.management;

import java.util.Set;

import org.jboss.managed.api.ComponentType;
import org.jboss.managed.api.DeploymentTemplateInfo;
import org.jboss.managed.api.ManagedComponent;
import org.jboss.managed.api.ManagedDeployment;
import org.jboss.profileservice.spi.NoSuchDeploymentException;

/**
 * The management view plugin spi for querying profiles for the
 * deployment management object interface roots.
 * 
 * @author Scott.Stark@jboss.org
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public interface ManagementView
{
   /**
    * Load all active profiles in the ManagementView for future operations.  
    * This must be called before the other ManagementView operations may be used.
    * 
    * @return if the ManagementView was reloaded
    */
   boolean load();

   /**
    * Reload the current profiles managed deployments/objects.
    */
   void reload();

   /**
    * Get the names of all deployments.
    * 
    * @return the deployment names
    */
   Set<String> getDeploymentNames();

   /**
    * Get the names of the deployment that have the given deployment type.
    * 
    * @param type - the deployment type
    * @return the names
    */
   Set<String> getDeploymentNamesForType(String type);

   /**
    * Search for all deployment names with a name matching the regex expression.
    * 
    * @param regex - the regex to query deployment name 
    * @return the deployment name.
    * @throws NoSuchDeploymentException if no matches are found 
    */
   Set<String> getMatchingDeploymentName(String regex)
      throws NoSuchDeploymentException;
   
   /**
    * Search for all deployments matched by the matcher.
    * 
    * @param name - the name to input to the matcher
    * @param matcher - the matcher 
    * @return a possibly empty set of ManagedDeployment accepted by the matcher
    * @throws Exception
    */
   Set<ManagedDeployment> getMatchingDeployments(String name, NameMatcher<ManagedDeployment> matcher)
      throws NoSuchDeploymentException, Exception;

   /**
    * Get a deployment.
    * 
    * @param name the name
    * @return the deployment
    * @throws NoSuchDeploymentException if no matches are found 
    */
   ManagedDeployment getDeployment(String name)
      throws NoSuchDeploymentException;

   /**
    * Get the deployments of a type.
    * 
    * @param type - the deployment or module type.
    * @return the possibly empty set of deployment with the given type.
    * @throws Exception
    */
   Set<ManagedDeployment> getDeploymentsForType(String type)
      throws Exception;

   /**
    * Get the components of a type. 
    * 
    * @param type - the component type.
    * @return the possibly empty set of components with the given type.
    * @throws Exception
    */
   Set<ManagedComponent> getComponentsForType(ComponentType type)
      throws Exception;

   /**
    * Obtain the ManagedComponent for the given name/type
    * 
    * @see #getComponent(String, ComponentType, NameMatcher)
    * @param name - the ManagedComponent#getName value to match
    * @param type - the component type
    * @return the possibly null ManagedComponent
    * @throws Exception
    */
   ManagedComponent getComponent(String name, ComponentType type)
      throws Exception;
   /**
    * Obtain the ManagedComponents for the given name/type and matcher.
    * 
    * @param name - the ManagedComponent#getName value to match
    * @param type - the component type
    * @param matcher - a matcher that compares name to candidate ManagedComponent
    * @return the possibly empty set of ManagedComponent matches.
    * @throws Exception
    */
   Set<ManagedComponent> getMatchingComponents(String name, ComponentType type,
         NameMatcher<ManagedComponent> matcher)
      throws Exception;

   /**
    * Update the given component. This takes the component ManagedDeployment,
    * obtains the ManagedObject set for the underlying deployment, and applies
    * the component ManagedProperty set to matching the deployment ManagedObject
    * properties.
    * 
    * @param comp the managed component
    * @throws Exception
    */
   void updateComponent(ManagedComponent comp)
      throws Exception;
   
   /**
    * Removes a given component. This will remove a component from the
    * associated ManagedDeployment.
    * 
    * @param comp the managed component.
    * @throws Exception
    */
   void removeComponent(ManagedComponent comp)
      throws Exception;
   
   /**
    * Get the registered DeploymentTemplate names.
    * 
    * TODO: probably needs a Map<String, DeploymentType> notion
    * @return the template names
    */
   Set<String> getTemplateNames();

   /**
    * Get a deployment template.
    * 
    * @param name - the deployment name to identify the template to retrieve
    * @return the named DeploymentTemplate
    * @throws NoSuchDeploymentException - if there is no such template
    */
   DeploymentTemplateInfo getTemplate(String name)
      throws NoSuchDeploymentException;

   /**
    * Create a deployment based on the given DeploymentTemplateInfo.
    * 
    * @param deploymentBaseName the base name
    * @param info the deployment template info
    * @throws Exception
    */
   void applyTemplate(String deploymentBaseName, DeploymentTemplateInfo info)
       throws Exception;

   /**
    * Process the changes made to the profile.
    * 
    * @throws Exception
    */
   void process() throws Exception;

}
