/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.profileservice.spi;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.Set;

import org.jboss.virtual.VirtualFile;

/**
 * An interface for managing the contents of a Profile.
 * 
 * @author Scott.Stark@jboss.org
 * @author adrian@jboss.org
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision: 85287 $
 */
public interface DeploymentRepository
{
   /**
    * Get the uris of this repository.
    * 
    * @return URI[] the uris of this repository
    */
   URI[] getRepositoryURIs();
	
   /**
    * Create the repository.
    * 
    * @throws Exception
    */
   void create() throws Exception;
   
   /**
    * Load the repository contents.
    * 
    * @throws Exception
    */
   void load() throws Exception;
   
   /**
    * Unload the repository;
    */
   void unload();
   
   /**
    * Delete the repository.
    * 
    * @throws Exception
    */
   void remove() throws Exception;
   
   /**
    * Get the time any contents were last modified.
    * 
    * @return last modified
    */
   long getLastModified();

   /**
    * Get the names of all deployments in the repository
    * 
    * @return the deployment names
    */
   Set<String> getDeploymentNames();
   
   /**
    * Get repository names.
    *
    * @param names the names
    * @return the matching repository names
    * @throws Exception for any error
    */
   String[] getRepositoryNames(String... names) throws Exception;

   /**
    * Upload raw deployment content to a profile repository. This does not make
    * the deployment available to getDeployments or getModifiedDeployments
    * 
    * @param name - the full vfs path of the deployment 
    * @param contentIS - the input stream for the deployment contents
    * @return the unique name of the deployment in the repository
    * @throws IOException
    */
   String addDeploymentContent(String name, InputStream contentIS)
      throws IOException;

   /**
    * Get the repository virtual file for the given deployment name.
    * 
    * @param name - the unique virtual file URI name as returned by addDeploymentContent
    * @return the deployment content virtual file
    * @throws IOException for any error
    */
   VirtualFile getDeploymentContent(String name)
      throws IOException, URISyntaxException;

   /**
    * lock deployment content and exclude it from modified deployment checks.
    * 
    * @param name - the full vfs path of the deployment
    * @return the deployment content flags, {@linkplain DeploymentContentFlags}
    */
   int lockDeploymentContent(String name);
   /**
    * Unlock a previously locked deployment content.
    * 
    * @param name - the full vfs path of the deployment
    * @return the deployment content flags, {@linkplain DeploymentContentFlags}
    */
   int unlockDeploymentContent(String name);

   /**
    * Get the status flags for the deployment path
    * 
    * @param name - the full vfs path of the deployment
    * @return the deployment content flags, {@linkplain DeploymentContentFlags}
    */
   int getDeploymentContentFlags(String name);
   /**
    * Set one or more flags for the deployment.
    * @see #clearDeploymentContentFlags(String)
    * 
    * @param name - the full vfs path of the deployment
    * @param flags - the deployment content flags, {@linkplain DeploymentContentFlags}
    */
   int setDeploymentContentFlags(String name, int flags);
   /**
    * Clear one or more flags for the deployment. This ands the compliment of
    * the flags argument into the existing content flags and returns the result.
    * 
    * @param name - the full vfs path of the deployment
    * @param flags - the deployment content flags, {@linkplain DeploymentContentFlags}
    */
   int clearDeploymentContentFlags(String name, int flags);
   /**
    * Does a deployment content have the indicated flag.
    * 
    * @param name - the full vfs path of the deployment
    * @param flags - the deployment content flags, {@linkplain DeploymentContentFlags}
    * @return true if the content flags contains the flag, false otherwise.
    */
   boolean hasDeploymentContentFlags(String name, int flag);

   /**
    * Add a deployment.
    * 
    * @param name - the name of the deployment
    * @param deployment the deployment
    * @throws Exception for any error
    */
   void addDeployment(String name, ProfileDeployment deployment)
      throws Exception;

   /**
    * Get a named deployment.
    * 
    * @param name - the deployment name
    * @return the deployment
    * @throws NoSuchDeploymentException - if there is no such deployment
    */
   ProfileDeployment getDeployment(String name)
      throws NoSuchDeploymentException;

   /**
    * Remove a deployment.
    * 
    * @param name - the name of the deployment
    * @return the removed deployment
    * @throws Exception for any error
    */
   ProfileDeployment removeDeployment(String name) throws Exception;

   /**
    * Get all deployments defined in this profile
    * 
    * @return the deployment instances in this profile
    */
   Collection<ProfileDeployment> getDeployments();

   /**
    * Get the modified deployments.
    * 
    * @return the modified deployments, empty if there are no
    *  modifications or hot deployment is not supported.
    * @throws Exception for any error
    */
   Collection<ModificationInfo> getModifiedDeployments() throws Exception;

}
